from django.core.management.base import BaseCommand
from django.contrib.auth.models import Group, Permission
from django.contrib.contenttypes.models import ContentType


class Command(BaseCommand):
    help = 'Cria grupos de usuários e define permissões'

    def handle(self, *args, **options):
        self.stdout.write('Configurando grupos e permissões...')
        
        # Criar grupos
        gerentes_group, created = Group.objects.get_or_create(name='Gerentes')
        if created:
            self.stdout.write(self.style.SUCCESS('Grupo "Gerentes" criado'))
        
        colaboradores_group, created = Group.objects.get_or_create(name='Colaboradores')
        if created:
            self.stdout.write(self.style.SUCCESS('Grupo "Colaboradores" criado'))
        
        clientes_group, created = Group.objects.get_or_create(name='Clientes')
        if created:
            self.stdout.write(self.style.SUCCESS('Grupo "Clientes" criado'))
        
        # Configurar permissões para Gerentes
        self.setup_managers_permissions(gerentes_group)
        
        # Configurar permissões para Colaboradores
        self.setup_collaborators_permissions(colaboradores_group)
        
        # Configurar permissões para Clientes
        self.setup_clients_permissions(clientes_group)
        
        self.stdout.write(self.style.SUCCESS('Configuração concluída!'))
    
    def setup_managers_permissions(self, group):
        """Configura permissões para gerentes."""
        self.stdout.write('Configurando permissões para Gerentes...')
        
        # Gerentes podem fazer tudo exceto gerenciar usuários
        apps_models = [
            ('client', 'client'),
            ('collaborator', 'area'),
            ('collaborator', 'collaborator'),
            ('workspace', 'workspace'),
            ('portfolio', 'portfolio'),
            ('project', 'project'),
            ('board', 'board'),
            ('task', 'group'),
            ('task', 'task'),
            ('notification', 'notification'),
        ]
        
        permissions = []
        for app_label, model_name in apps_models:
            try:
                content_type = ContentType.objects.get(app_label=app_label, model=model_name)
                model_permissions = Permission.objects.filter(content_type=content_type)
                permissions.extend(model_permissions)
            except ContentType.DoesNotExist:
                self.stdout.write(
                    self.style.WARNING(f'ContentType não encontrado: {app_label}.{model_name}')
                )
        
        group.permissions.set(permissions)
        self.stdout.write(f'Adicionadas {len(permissions)} permissões para Gerentes')
    
    def setup_collaborators_permissions(self, group):
        """Configura permissões para colaboradores."""
        self.stdout.write('Configurando permissões para Colaboradores...')
        
        # Colaboradores podem visualizar a maioria das coisas e editar suas tarefas
        view_permissions = []
        edit_permissions = []
        
        # Permissões de visualização
        apps_models = [
            ('client', 'client'),
            ('collaborator', 'area'),
            ('collaborator', 'collaborator'),
            ('workspace', 'workspace'),
            ('portfolio', 'portfolio'),
            ('project', 'project'),
            ('board', 'board'),
            ('task', 'group'),
            ('task', 'task'),
            ('notification', 'notification'),
        ]
        
        for app_label, model_name in apps_models:
            try:
                content_type = ContentType.objects.get(app_label=app_label, model=model_name)
                
                # Permissão de visualização
                view_perm = Permission.objects.filter(
                    content_type=content_type,
                    codename__startswith='view_'
                )
                view_permissions.extend(view_perm)
                
                # Permissões específicas para tarefas e notificações
                if model_name in ['task', 'notification']:
                    change_perm = Permission.objects.filter(
                        content_type=content_type,
                        codename__startswith='change_'
                    )
                    edit_permissions.extend(change_perm)
                
            except ContentType.DoesNotExist:
                self.stdout.write(
                    self.style.WARNING(f'ContentType não encontrado: {app_label}.{model_name}')
                )
        
        all_permissions = view_permissions + edit_permissions
        group.permissions.set(all_permissions)
        self.stdout.write(f'Adicionadas {len(all_permissions)} permissões para Colaboradores')
    
    def setup_clients_permissions(self, group):
        """Configura permissões para clientes."""
        self.stdout.write('Configurando permissões para Clientes...')
        
        # Clientes podem apenas visualizar seus próprios projetos
        view_permissions = []
        
        apps_models = [
            ('workspace', 'workspace'),
            ('portfolio', 'portfolio'),
            ('project', 'project'),
            ('board', 'board'),
            ('task', 'group'),
            ('task', 'task'),
        ]
        
        for app_label, model_name in apps_models:
            try:
                content_type = ContentType.objects.get(app_label=app_label, model=model_name)
                view_perm = Permission.objects.filter(
                    content_type=content_type,
                    codename__startswith='view_'
                )
                view_permissions.extend(view_perm)
            except ContentType.DoesNotExist:
                self.stdout.write(
                    self.style.WARNING(f'ContentType não encontrado: {app_label}.{model_name}')
                )
        
        group.permissions.set(view_permissions)
        self.stdout.write(f'Adicionadas {len(view_permissions)} permissões para Clientes')

