from django.shortcuts import redirect
from django.urls import reverse
from django.contrib import messages
from django.utils.deprecation import MiddlewareMixin


class CollaboratorRequiredMiddleware(MiddlewareMixin):
    """
    Middleware que verifica se o usuário autenticado é um colaborador ativo.
    """
    
    # URLs que não precisam de verificação de colaborador
    EXEMPT_URLS = [
        '/admin/',
        '/accounts/',
        '/api/',
        '/static/',
        '/media/',
    ]
    
    def process_request(self, request):
        # Pular verificação para URLs isentas
        for exempt_url in self.EXEMPT_URLS:
            if request.path.startswith(exempt_url):
                return None
        
        # Pular verificação para usuários não autenticados
        if not request.user.is_authenticated:
            return None
        
        # Superusuários podem acessar tudo
        if request.user.is_superuser:
            return None
        
        # Verificar se o usuário tem um colaborador ativo
        try:
            collaborator = request.user.collaborator
            if not collaborator.ativo:
                messages.error(request, 'Sua conta de colaborador está inativa. Entre em contato com o administrador.')
                return redirect('admin:index')
        except:
            # Usuário não tem colaborador associado
            if not request.user.is_staff:
                messages.error(request, 'Você não tem um perfil de colaborador associado. Entre em contato com o administrador.')
                return redirect('admin:index')
        
        return None


class ProjectAccessMiddleware(MiddlewareMixin):
    """
    Middleware que controla o acesso a projetos baseado nas permissões do usuário.
    """
    
    def process_request(self, request):
        # Aplicar apenas para URLs de projetos, boards e tarefas
        project_urls = ['/projects/', '/boards/', '/tasks/']
        
        if not any(request.path.startswith(url) for url in project_urls):
            return None
        
        # Pular verificação para usuários não autenticados (será tratado pelo login_required)
        if not request.user.is_authenticated:
            return None
        
        # Superusuários e gerentes podem acessar tudo
        if request.user.is_superuser or request.user.groups.filter(name='Gerentes').exists():
            return None
        
        # Para colaboradores, verificar se têm acesso ao projeto específico
        # Esta verificação será feita nas views individuais
        return None


class AuditMiddleware(MiddlewareMixin):
    """
    Middleware para auditoria de ações dos usuários.
    """
    
    def process_request(self, request):
        # Adicionar informações de auditoria ao request
        if request.user.is_authenticated:
            request.audit_user = request.user
            request.audit_ip = self.get_client_ip(request)
        
        return None
    
    def get_client_ip(self, request):
        """Obtém o IP real do cliente."""
        x_forwarded_for = request.META.get('HTTP_X_FORWARDED_FOR')
        if x_forwarded_for:
            ip = x_forwarded_for.split(',')[0]
        else:
            ip = request.META.get('REMOTE_ADDR')
        return ip


class TimezoneMiddleware(MiddlewareMixin):
    """
    Middleware para definir timezone baseado no colaborador.
    """
    
    def process_request(self, request):
        if request.user.is_authenticated:
            try:
                # Usar timezone do colaborador se disponível
                collaborator = request.user.collaborator
                # Por enquanto usar timezone padrão, mas pode ser expandido
                # para permitir que colaboradores definam seu timezone
                pass
            except:
                pass
        
        return None

