from rest_framework import permissions
from django.contrib.auth.models import Group


class IsOwnerOrReadOnly(permissions.BasePermission):
    """
    Permissão customizada para permitir apenas aos donos de um objeto editá-lo.
    """

    def has_object_permission(self, request, view, obj):
        # Permissões de leitura são permitidas para qualquer request,
        # então sempre permitimos requests GET, HEAD ou OPTIONS.
        if request.method in permissions.SAFE_METHODS:
            return True

        # Permissões de escrita são apenas para o dono do objeto.
        return obj.owner == request.user


class IsCollaboratorOrReadOnly(permissions.BasePermission):
    """
    Permissão para colaboradores.
    """

    def has_permission(self, request, view):
        if request.method in permissions.SAFE_METHODS:
            return True
        
        # Verifica se o usuário é um colaborador
        try:
            return hasattr(request.user, 'collaborator') and request.user.collaborator.ativo
        except:
            return False


class IsManagerOrReadOnly(permissions.BasePermission):
    """
    Permissão para gerentes.
    """

    def has_permission(self, request, view):
        if request.method in permissions.SAFE_METHODS:
            return True
        
        # Verifica se o usuário está no grupo de gerentes
        return request.user.groups.filter(name='Gerentes').exists()


class IsAdminOrReadOnly(permissions.BasePermission):
    """
    Permissão para administradores.
    """

    def has_permission(self, request, view):
        if request.method in permissions.SAFE_METHODS:
            return True
        
        return request.user.is_staff or request.user.is_superuser


class CanManageProject(permissions.BasePermission):
    """
    Permissão para gerenciar projetos.
    """

    def has_object_permission(self, request, view, obj):
        # Administradores podem tudo
        if request.user.is_superuser:
            return True
        
        # Gerentes podem gerenciar projetos
        if request.user.groups.filter(name='Gerentes').exists():
            return True
        
        # Colaboradores podem apenas visualizar
        if request.method in permissions.SAFE_METHODS:
            try:
                return hasattr(request.user, 'collaborator') and request.user.collaborator.ativo
            except:
                return False
        
        return False


class CanManageTask(permissions.BasePermission):
    """
    Permissão para gerenciar tarefas.
    """

    def has_object_permission(self, request, view, obj):
        # Administradores podem tudo
        if request.user.is_superuser:
            return True
        
        # Gerentes podem gerenciar tarefas
        if request.user.groups.filter(name='Gerentes').exists():
            return True
        
        # Responsável pela tarefa pode editá-la
        try:
            if hasattr(request.user, 'collaborator') and obj.responsavel == request.user.collaborator:
                return True
        except:
            pass
        
        # Colaboradores podem apenas visualizar
        if request.method in permissions.SAFE_METHODS:
            try:
                return hasattr(request.user, 'collaborator') and request.user.collaborator.ativo
            except:
                return False
        
        return False


class CanViewClient(permissions.BasePermission):
    """
    Permissão para visualizar clientes.
    """

    def has_permission(self, request, view):
        # Apenas usuários autenticados
        if not request.user.is_authenticated:
            return False
        
        # Administradores e gerentes podem tudo
        if request.user.is_superuser or request.user.groups.filter(name='Gerentes').exists():
            return True
        
        # Colaboradores podem apenas visualizar
        if request.method in permissions.SAFE_METHODS:
            try:
                return hasattr(request.user, 'collaborator') and request.user.collaborator.ativo
            except:
                return False
        
        return False


def user_can_access_workspace(user, workspace):
    """
    Verifica se o usuário pode acessar um workspace específico.
    """
    if user.is_superuser:
        return True
    
    if user.groups.filter(name='Gerentes').exists():
        return True
    
    # Colaboradores podem acessar workspaces onde têm tarefas
    try:
        collaborator = user.collaborator
        if not collaborator.ativo:
            return False
        
        # Verifica se tem tarefas no workspace
        from task.models import Task
        return Task.objects.filter(
            responsavel=collaborator,
            group__board__project__workspace=workspace
        ).exists()
    except:
        return False


def user_can_access_project(user, project):
    """
    Verifica se o usuário pode acessar um projeto específico.
    """
    if user.is_superuser:
        return True
    
    if user.groups.filter(name='Gerentes').exists():
        return True
    
    # Colaboradores podem acessar projetos onde têm tarefas
    try:
        collaborator = user.collaborator
        if not collaborator.ativo:
            return False
        
        # Verifica se tem tarefas no projeto
        from task.models import Task
        return Task.objects.filter(
            responsavel=collaborator,
            group__board__project=project
        ).exists()
    except:
        return False


def get_user_workspaces(user):
    """
    Retorna os workspaces que o usuário pode acessar.
    """
    from workspace.models import Workspace
    
    if user.is_superuser or user.groups.filter(name='Gerentes').exists():
        return Workspace.objects.filter(ativo=True)
    
    try:
        collaborator = user.collaborator
        if not collaborator.ativo:
            return Workspace.objects.none()
        
        # Workspaces onde o colaborador tem tarefas
        from task.models import Task
        workspace_ids = Task.objects.filter(
            responsavel=collaborator
        ).values_list('group__board__project__workspace_id', flat=True).distinct()
        
        return Workspace.objects.filter(id__in=workspace_ids, ativo=True)
    except:
        return Workspace.objects.none()


def get_user_projects(user):
    """
    Retorna os projetos que o usuário pode acessar.
    """
    from project.models import Project
    
    if user.is_superuser or user.groups.filter(name='Gerentes').exists():
        return Project.objects.filter(ativo=True)
    
    try:
        collaborator = user.collaborator
        if not collaborator.ativo:
            return Project.objects.none()
        
        # Projetos onde o colaborador tem tarefas
        from task.models import Task
        project_ids = Task.objects.filter(
            responsavel=collaborator
        ).values_list('group__board__project_id', flat=True).distinct()
        
        return Project.objects.filter(id__in=project_ids, ativo=True)
    except:
        return Project.objects.none()

