from django.shortcuts import render
from django.contrib.auth.decorators import login_required
from django.db.models import Count, Q
from django.utils import timezone
from datetime import timedelta
from client.models import Client
from collaborator.models import Collaborator
from project.models import Project
from task.models import Task, Group
from board.models import Board
from workspace.models import Workspace
from system_logs.utils import log_system_action


@login_required
def home(request):
    """
    View para a página inicial/dashboard.
    """
    
    # Registra acesso ao dashboard
    log_system_action(
        request=request,
        action='VIEW',
        description=f"Acesso ao dashboard por {request.user.username}",
        severity='LOW'
    )
    
    # Estatísticas gerais
    stats = {
        'total_clients': Client.objects.filter(ativo=True).count(),
        'total_projects': Project.objects.filter(ativo=True).count(),
        'total_tasks': Task.objects.count(),
        'total_collaborators': Collaborator.objects.filter(ativo=True).count(),
        'total_workspaces': Workspace.objects.filter(ativo=True).count(),
        'total_boards': Board.objects.filter(ativo=True).count(),
        'total_groups': Group.objects.filter(ativo=True).count(),
    }
    
    # Estatísticas de tarefas
    task_stats = {
        'tasks_nao_iniciadas': Task.objects.filter(status='nao_iniciada').count(),
        'tasks_em_andamento': Task.objects.filter(status='em_andamento').count(),
        'tasks_concluidas': Task.objects.filter(status='concluida').count(),
        'tasks_atrasadas': Task.objects.filter(
            Q(status__in=['nao_iniciada', 'em_andamento', 'pausada']) &
            Q(cronograma_fim__lt=timezone.now().date())
        ).count(),
    }
    
    # Projetos recentes
    recent_projects = Project.objects.filter(ativo=True).select_related(
        'workspace', 'cliente', 'portfolio'
    ).order_by('-created_at')[:5]
    
    # Tarefas pendentes (do usuário logado se for colaborador)
    pending_tasks = Task.objects.filter(
        status__in=['nao_iniciada', 'em_andamento', 'pausada']
    ).select_related(
        'group__board__project', 'responsavel'
    ).order_by('cronograma_fim')[:5]
    
    try:
        # Se o usuário é um colaborador, mostra apenas suas tarefas
        collaborator = request.user.collaborator
        pending_tasks = pending_tasks.filter(responsavel=collaborator)
    except:
        # Se não é colaborador, mostra todas as tarefas
        pass
    
    # Projetos por status
    projects_by_status = Project.objects.filter(ativo=True).values('status').annotate(
        count=Count('id')
    ).order_by('status')
    
    # Tarefas por prioridade
    tasks_by_priority = Task.objects.values('prioridade').annotate(
        count=Count('id')
    ).order_by('prioridade')
    
    # Tarefas vencendo hoje
    today = timezone.now().date()
    tasks_due_today = Task.objects.filter(
        cronograma_fim=today,
        status__in=['nao_iniciada', 'em_andamento', 'pausada']
    ).select_related('group__board__project')[:3]
    
    # Tarefas vencendo esta semana
    week_end = today + timedelta(days=7)
    tasks_due_this_week = Task.objects.filter(
        cronograma_fim__range=[today, week_end],
        status__in=['nao_iniciada', 'em_andamento', 'pausada']
    ).select_related('group__board__project')[:3]
    
    context = {
        'stats': stats,
        'task_stats': task_stats,
        'recent_projects': recent_projects,
        'pending_tasks': pending_tasks,
        'projects_by_status': projects_by_status,
        'tasks_by_priority': tasks_by_priority,
        'tasks_due_today': tasks_due_today,
        'tasks_due_this_week': tasks_due_this_week,
    }
    
    return render(request, 'home.html', context)

