from django.contrib import admin
from .models import Notification


@admin.register(Notification)
class NotificationAdmin(admin.ModelAdmin):
    """
    Admin para modelo Notification.
    """
    list_display = ['titulo', 'destinatario', 'tipo', 'lida', 'status', 'created_at']
    list_filter = ['tipo', 'lida', 'status', 'created_at', 'enviada_em']
    search_fields = ['titulo', 'mensagem', 'destinatario__nome', 'tarefa__nome']
    list_editable = ['lida', 'status']
    ordering = ['-created_at']
    date_hierarchy = 'created_at'
    
    fieldsets = (
        ('Informações Básicas', {
            'fields': ('titulo', 'mensagem', 'destinatario', 'tipo')
        }),
        ('Relacionamentos', {
            'fields': ('tarefa',)
        }),
        ('Status', {
            'fields': ('lida', 'status', 'lida_em', 'enviada_em')
        }),
        ('Metadados', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    
    readonly_fields = ['created_at', 'updated_at', 'lida_em', 'enviada_em']
    
    def get_queryset(self, request):
        return super().get_queryset(request).select_related('destinatario', 'tarefa')
    
    actions = ['mark_as_read', 'mark_as_unread', 'resend_notifications']
    
    def mark_as_read(self, request, queryset):
        """Marca notificações como lidas."""
        count = 0
        for notification in queryset:
            if not notification.lida:
                notification.marcar_como_lida()
                count += 1
        
        self.message_user(request, f'{count} notificações marcadas como lidas.')
    mark_as_read.short_description = 'Marcar como lidas'
    
    def mark_as_unread(self, request, queryset):
        """Marca notificações como não lidas."""
        count = queryset.filter(lida=True).update(lida=False, lida_em=None)
        self.message_user(request, f'{count} notificações marcadas como não lidas.')
    mark_as_unread.short_description = 'Marcar como não lidas'
    
    def resend_notifications(self, request, queryset):
        """Reenvia notificações por email."""
        count = 0
        for notification in queryset:
            notification.enviar_email()
            count += 1
        
        self.message_user(request, f'{count} notificações reenviadas.')
    resend_notifications.short_description = 'Reenviar por email'

