from rest_framework import viewsets, status
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter

from .models import Notification
from .serializers import (
    NotificationSerializer, 
    NotificationDetailSerializer, 
    NotificationListSerializer,
    NotificationCreateSerializer,
    NotificationMarkAsReadSerializer,
    NotificationBulkMarkAsReadSerializer
)


class NotificationViewSet(viewsets.ModelViewSet):
    """
    ViewSet para operações CRUD de notificações.
    """
    
    queryset = Notification.objects.all()
    permission_classes = [IsAuthenticated]
    filter_backends = [DjangoFilterBackend, SearchFilter, OrderingFilter]
    filterset_fields = ['tipo', 'status', 'lida', 'destinatario']
    search_fields = ['titulo', 'mensagem', 'destinatario__nome', 'tarefa__nome']
    ordering_fields = ['created_at', 'enviada_em', 'lida_em']
    ordering = ['-created_at']

    def get_serializer_class(self):
        if self.action == 'create':
            return NotificationCreateSerializer
        if self.action == 'retrieve':
            return NotificationDetailSerializer
        if self.action == 'list':
            return NotificationListSerializer
        if self.action == 'mark_as_read':
            return NotificationMarkAsReadSerializer
        if self.action == 'bulk_mark_as_read':
            return NotificationBulkMarkAsReadSerializer
        return NotificationSerializer

    def get_queryset(self):
        queryset = Notification.objects.select_related('destinatario', 'tarefa').all()
        
        # Filtro por destinatário (apenas suas próprias notificações)
        if not self.request.user.is_superuser:
            try:
                collaborator = self.request.user.collaborator
                queryset = queryset.filter(destinatario=collaborator)
            except:
                queryset = queryset.none()
        
        # Filtro por status de leitura
        lida = self.request.query_params.get('lida', None)
        if lida is not None:
            queryset = queryset.filter(lida=lida.lower() == 'true')
            
        # Filtro por tipo
        tipo = self.request.query_params.get('tipo', None)
        if tipo:
            queryset = queryset.filter(tipo=tipo)
            
        # Filtro por status de envio
        status_envio = self.request.query_params.get('status', None)
        if status_envio:
            queryset = queryset.filter(status=status_envio)
            
        return queryset

    @action(detail=True, methods=['patch'])
    def mark_as_read(self, request, pk=None):
        """
        Marca uma notificação como lida.
        """
        notification = self.get_object()
        
        # Verifica se o usuário pode marcar esta notificação
        if not request.user.is_superuser:
            try:
                collaborator = request.user.collaborator
                if notification.destinatario != collaborator:
                    return Response(
                        {"error": "Você não tem permissão para marcar esta notificação"}, 
                        status=status.HTTP_403_FORBIDDEN
                    )
            except:
                return Response(
                    {"error": "Colaborador não encontrado"}, 
                    status=status.HTTP_404_NOT_FOUND
                )
        
        notification.marcar_como_lida()
        serializer = self.get_serializer(notification)
        return Response(serializer.data)

    @action(detail=False, methods=['post'])
    def bulk_mark_as_read(self, request):
        """
        Marca múltiplas notificações como lidas.
        """
        serializer = NotificationBulkMarkAsReadSerializer(data=request.data)
        
        if serializer.is_valid():
            notification_ids = serializer.validated_data['notification_ids']
            
            # Filtra apenas as notificações do usuário
            queryset = self.get_queryset().filter(id__in=notification_ids, lida=False)
            
            # Marca todas como lidas
            for notification in queryset:
                notification.marcar_como_lida()
            
            return Response({"message": f"{queryset.count()} notificações marcadas como lidas"})
        
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    @action(detail=False, methods=['post'])
    def mark_all_as_read(self, request):
        """
        Marca todas as notificações não lidas do usuário como lidas.
        """
        queryset = self.get_queryset().filter(lida=False)
        
        count = 0
        for notification in queryset:
            notification.marcar_como_lida()
            count += 1
        
        return Response({"message": f"{count} notificações marcadas como lidas"})

    @action(detail=True, methods=['post'])
    def resend(self, request, pk=None):
        """
        Reenvia uma notificação por email.
        """
        notification = self.get_object()
        
        # Apenas admins ou o próprio destinatário podem reenviar
        if not request.user.is_superuser:
            try:
                collaborator = request.user.collaborator
                if notification.destinatario != collaborator:
                    return Response(
                        {"error": "Você não tem permissão para reenviar esta notificação"}, 
                        status=status.HTTP_403_FORBIDDEN
                    )
            except:
                return Response(
                    {"error": "Colaborador não encontrado"}, 
                    status=status.HTTP_404_NOT_FOUND
                )
        
        notification.enviar_email()
        serializer = self.get_serializer(notification)
        return Response(serializer.data)

    @action(detail=False, methods=['get'])
    def unread_count(self, request):
        """
        Retorna a contagem de notificações não lidas do usuário.
        """
        queryset = self.get_queryset().filter(lida=False)
        count = queryset.count()
        
        return Response({"unread_count": count})

    @action(detail=False, methods=['get'])
    def statistics(self, request):
        """
        Retorna estatísticas das notificações do usuário.
        """
        queryset = self.get_queryset()
        
        stats = {
            'total': queryset.count(),
            'lidas': queryset.filter(lida=True).count(),
            'nao_lidas': queryset.filter(lida=False).count(),
            'por_tipo': {},
            'por_status': {},
        }
        
        # Estatísticas por tipo
        for notification in queryset:
            tipo_key = notification.get_tipo_display()
            if tipo_key not in stats['por_tipo']:
                stats['por_tipo'][tipo_key] = 0
            stats['por_tipo'][tipo_key] += 1
            
            status_key = notification.get_status_display()
            if status_key not in stats['por_status']:
                stats['por_status'][status_key] = 0
            stats['por_status'][status_key] += 1
        
        return Response(stats)

