from django.db import models
from django.contrib.auth.models import User
from django.core.mail import send_mail
from django.conf import settings
from collaborator.models import Collaborator
from task.models import Task


class Notification(models.Model):
    """
    Modelo para representar notificações do sistema.
    """
    
    TIPO_CHOICES = [
        ('responsavel_adicionado', 'Responsável Adicionado'),
        ('tarefa_atrasada', 'Tarefa Atrasada'),
        ('tarefa_concluida', 'Tarefa Concluída'),
        ('prazo_proximo', 'Prazo Próximo'),
        ('dependencia_concluida', 'Dependência Concluída'),
    ]
    
    STATUS_CHOICES = [
        ('pendente', 'Pendente'),
        ('enviada', 'Enviada'),
        ('erro', 'Erro'),
    ]
    
    tipo = models.CharField(
        max_length=30,
        choices=TIPO_CHOICES,
        verbose_name="Tipo",
        help_text="Tipo da notificação"
    )
    
    destinatario = models.ForeignKey(
        Collaborator,
        on_delete=models.CASCADE,
        verbose_name="Destinatário",
        help_text="Colaborador que receberá a notificação",
        related_name="notifications"
    )
    
    tarefa = models.ForeignKey(
        Task,
        on_delete=models.CASCADE,
        verbose_name="Tarefa",
        help_text="Tarefa relacionada à notificação",
        related_name="notifications"
    )
    
    titulo = models.CharField(
        max_length=200,
        verbose_name="Título",
        help_text="Título da notificação"
    )
    
    mensagem = models.TextField(
        verbose_name="Mensagem",
        help_text="Conteúdo da notificação"
    )
    
    status = models.CharField(
        max_length=10,
        choices=STATUS_CHOICES,
        default='pendente',
        verbose_name="Status",
        help_text="Status da notificação"
    )
    
    enviada_em = models.DateTimeField(
        blank=True,
        null=True,
        verbose_name="Enviada em",
        help_text="Data e hora do envio"
    )
    
    erro_envio = models.TextField(
        blank=True,
        null=True,
        verbose_name="Erro de Envio",
        help_text="Detalhes do erro caso o envio falhe"
    )
    
    lida = models.BooleanField(
        default=False,
        verbose_name="Lida",
        help_text="Indica se a notificação foi lida"
    )
    
    lida_em = models.DateTimeField(
        blank=True,
        null=True,
        verbose_name="Lida em",
        help_text="Data e hora da leitura"
    )
    
    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name="Criado em"
    )
    
    updated_at = models.DateTimeField(
        auto_now=True,
        verbose_name="Atualizado em"
    )

    class Meta:
        verbose_name = "Notificação"
        verbose_name_plural = "Notificações"
        ordering = ['-created_at']

    def __str__(self):
        return f"{self.get_tipo_display()} - {self.destinatario.nome}"

    def get_absolute_url(self):
        from django.urls import reverse
        return reverse('notification:detail', kwargs={'pk': self.pk})

    def enviar_email(self):
        """Envia a notificação por email."""
        try:
            from django.utils import timezone
            
            subject = self.titulo
            message = self.mensagem
            from_email = settings.DEFAULT_FROM_EMAIL
            recipient_list = [self.destinatario.email]
            
            send_mail(
                subject=subject,
                message=message,
                from_email=from_email,
                recipient_list=recipient_list,
                fail_silently=False
            )
            
            self.status = 'enviada'
            self.enviada_em = timezone.now()
            self.erro_envio = None
            
        except Exception as e:
            self.status = 'erro'
            self.erro_envio = str(e)
        
        self.save()

    def marcar_como_lida(self):
        """Marca a notificação como lida."""
        from django.utils import timezone
        self.lida = True
        self.lida_em = timezone.now()
        self.save()

    @classmethod
    def criar_notificacao_responsavel_adicionado(cls, tarefa, responsavel):
        """Cria notificação quando um responsável é adicionado à tarefa."""
        titulo = f"Você foi designado para a tarefa: {tarefa.nome}"
        mensagem = f"""
        Olá {responsavel.nome},
        
        Você foi designado como responsável pela tarefa "{tarefa.nome}".
        
        Detalhes da tarefa:
        - Projeto: {tarefa.group.board.project.nome}
        - Quadro: {tarefa.group.board.nome}
        - Grupo: {tarefa.group.nome}
        - Prazo: {tarefa.cronograma_fim.strftime('%d/%m/%Y')}
        - Prioridade: {tarefa.get_prioridade_display()}
        
        Acesse o sistema para mais detalhes.
        """
        
        notification = cls.objects.create(
            tipo='responsavel_adicionado',
            destinatario=responsavel,
            tarefa=tarefa,
            titulo=titulo,
            mensagem=mensagem
        )
        
        notification.enviar_email()
        return notification

    @classmethod
    def criar_notificacao_tarefa_atrasada(cls, tarefa):
        """Cria notificação quando uma tarefa está atrasada."""
        if not tarefa.responsavel:
            return None
            
        titulo = f"Tarefa atrasada: {tarefa.nome}"
        mensagem = f"""
        Olá {tarefa.responsavel.nome},
        
        A tarefa "{tarefa.nome}" está atrasada.
        
        Detalhes:
        - Prazo original: {tarefa.cronograma_fim.strftime('%d/%m/%Y')}
        - Dias de atraso: {tarefa.dias_atraso}
        - Status atual: {tarefa.get_status_display()}
        
        Por favor, atualize o status da tarefa ou entre em contato com seu gestor.
        """
        
        notification = cls.objects.create(
            tipo='tarefa_atrasada',
            destinatario=tarefa.responsavel,
            tarefa=tarefa,
            titulo=titulo,
            mensagem=mensagem
        )
        
        notification.enviar_email()
        return notification

    @classmethod
    def criar_notificacao_tarefa_concluida(cls, tarefa):
        """Cria notificação quando uma tarefa é concluída."""
        if not tarefa.responsavel:
            return None
            
        titulo = f"Tarefa concluída: {tarefa.nome}"
        mensagem = f"""
        Parabéns {tarefa.responsavel.nome}!
        
        A tarefa "{tarefa.nome}" foi marcada como concluída.
        
        Detalhes:
        - Data de conclusão: {tarefa.data_conclusao.strftime('%d/%m/%Y às %H:%M')}
        - Esforço utilizado: {tarefa.esforco_utilizado} horas
        - Esforço previsto: {tarefa.esforco_previsto} horas
        
        Obrigado pelo seu trabalho!
        """
        
        notification = cls.objects.create(
            tipo='tarefa_concluida',
            destinatario=tarefa.responsavel,
            tarefa=tarefa,
            titulo=titulo,
            mensagem=mensagem
        )
        
        notification.enviar_email()
        return notification

    @classmethod
    def criar_notificacao_prazo_proximo(cls, tarefa):
        """Cria notificação quando uma tarefa tem prazo próximo."""
        if not tarefa.responsavel:
            return None
            
        titulo = f"Prazo próximo: {tarefa.nome}"
        mensagem = f"""
        Olá {tarefa.responsavel.nome},
        
        A tarefa "{tarefa.nome}" vence em breve.
        
        Detalhes:
        - Prazo: {tarefa.cronograma_fim.strftime('%d/%m/%Y')}
        - Status atual: {tarefa.get_status_display()}
        - Prioridade: {tarefa.get_prioridade_display()}
        
        Por favor, verifique o progresso da tarefa e atualize o status se necessário.
        """
        
        notification = cls.objects.create(
            tipo='prazo_proximo',
            destinatario=tarefa.responsavel,
            tarefa=tarefa,
            titulo=titulo,
            mensagem=mensagem
        )
        
        notification.enviar_email()
        return notification

