from rest_framework import serializers
from .models import Notification
from collaborator.serializers import CollaboratorListSerializer
from task.serializers import TaskListSerializer


class NotificationSerializer(serializers.ModelSerializer):
    """
    Serializer para o modelo Notification.
    """
    
    destinatario_nome = serializers.CharField(source='destinatario.nome', read_only=True)
    tarefa_nome = serializers.CharField(source='tarefa.nome', read_only=True)
    project_nome = serializers.CharField(source='tarefa.group.board.project.nome', read_only=True)
    
    class Meta:
        model = Notification
        fields = [
            'id', 'tipo', 'destinatario', 'destinatario_nome', 'tarefa', 'tarefa_nome',
            'project_nome', 'titulo', 'mensagem', 'status', 'enviada_em', 'erro_envio',
            'lida', 'lida_em', 'created_at', 'updated_at'
        ]
        read_only_fields = [
            'id', 'status', 'enviada_em', 'erro_envio', 'lida_em',
            'created_at', 'updated_at'
        ]


class NotificationDetailSerializer(serializers.ModelSerializer):
    """
    Serializer detalhado para o modelo Notification.
    """
    
    destinatario = CollaboratorListSerializer(read_only=True)
    tarefa = TaskListSerializer(read_only=True)
    
    class Meta:
        model = Notification
        fields = [
            'id', 'tipo', 'destinatario', 'tarefa', 'titulo', 'mensagem',
            'status', 'enviada_em', 'erro_envio', 'lida', 'lida_em',
            'created_at', 'updated_at'
        ]
        read_only_fields = [
            'id', 'status', 'enviada_em', 'erro_envio', 'lida_em',
            'created_at', 'updated_at'
        ]


class NotificationListSerializer(serializers.ModelSerializer):
    """
    Serializer simplificado para listagem de notificações.
    """
    
    destinatario_nome = serializers.CharField(source='destinatario.nome', read_only=True)
    tarefa_nome = serializers.CharField(source='tarefa.nome', read_only=True)
    
    class Meta:
        model = Notification
        fields = [
            'id', 'tipo', 'destinatario_nome', 'tarefa_nome', 'titulo',
            'status', 'lida', 'created_at'
        ]


class NotificationCreateSerializer(serializers.ModelSerializer):
    """
    Serializer para criação de notificações.
    """
    
    class Meta:
        model = Notification
        fields = ['tipo', 'destinatario', 'tarefa', 'titulo', 'mensagem']


class NotificationMarkAsReadSerializer(serializers.Serializer):
    """
    Serializer para marcar notificação como lida.
    """
    
    mark_as_read = serializers.BooleanField(default=True)


class NotificationBulkMarkAsReadSerializer(serializers.Serializer):
    """
    Serializer para marcar múltiplas notificações como lidas.
    """
    
    notification_ids = serializers.ListField(
        child=serializers.IntegerField(),
        allow_empty=False
    )
    
    def validate_notification_ids(self, value):
        # Verifica se todas as notificações existem
        existing_ids = Notification.objects.filter(id__in=value).values_list('id', flat=True)
        missing_ids = set(value) - set(existing_ids)
        
        if missing_ids:
            raise serializers.ValidationError(
                f"Notificações não encontradas: {list(missing_ids)}"
            )
        
        return value

