from celery import shared_task
from django.core.mail import send_mail
from django.conf import settings
from django.template.loader import render_to_string
from django.utils.html import strip_tags
from django.utils import timezone
from .models import Notification


@shared_task
def enviar_notificacao_email(notification_id):
    """
    Tarefa assíncrona para enviar notificação por email.
    """
    try:
        notification = Notification.objects.get(id=notification_id)
        
        # Renderizar template de email
        html_message = render_to_string('notification/email.html', {
            'notification': notification,
            'site_name': 'Gestão de Projetos'
        })
        plain_message = strip_tags(html_message)
        
        # Enviar email
        send_mail(
            subject=f"[Gestão de Projetos] {notification.titulo}",
            message=plain_message,
            from_email=settings.DEFAULT_FROM_EMAIL,
            recipient_list=[notification.destinatario.email],
            html_message=html_message,
            fail_silently=False,
        )
        
        # Atualizar status da notificação
        notification.status = 'enviada'
        notification.enviada_em = timezone.now()
        notification.save()
        
        return f"Email enviado para {notification.destinatario.email}"
        
    except Notification.DoesNotExist:
        return f"Notificação {notification_id} não encontrada"
    except Exception as e:
        # Marcar como erro
        try:
            notification = Notification.objects.get(id=notification_id)
            notification.status = 'erro'
            notification.save()
        except:
            pass
        
        return f"Erro ao enviar email: {str(e)}"


@shared_task
def processar_notificacoes_pendentes():
    """
    Tarefa para processar notificações pendentes.
    """
    from django.utils import timezone
    
    # Buscar notificações pendentes
    notifications = Notification.objects.filter(
        status='pendente',
        created_at__lte=timezone.now()
    )
    
    count = 0
    for notification in notifications:
        enviar_notificacao_email.delay(notification.id)
        count += 1
    
    return f"Processadas {count} notificações"


@shared_task
def verificar_tarefas_atrasadas():
    """
    Tarefa para verificar tarefas atrasadas e criar notificações.
    """
    from django.utils import timezone
    from task.models import Task
    
    # Buscar tarefas atrasadas
    tarefas_atrasadas = Task.objects.filter(
        cronograma_fim__lt=timezone.now().date(),
        status__in=['nao_iniciada', 'em_andamento', 'pausada']
    )
    
    count = 0
    for task in tarefas_atrasadas:
        if task.responsavel:
            # Verificar se já existe notificação de atraso para esta tarefa
            existing = Notification.objects.filter(
                tipo='tarefa_atrasada',
                tarefa=task,
                destinatario=task.responsavel,
                created_at__date=timezone.now().date()
            ).exists()
            
            if not existing:
                Notification.criar_notificacao_tarefa_atrasada(task)
                count += 1
    
    return f"Criadas {count} notificações de tarefas atrasadas"


@shared_task
def verificar_prazos_proximos():
    """
    Tarefa para verificar tarefas com prazos próximos.
    """
    from django.utils import timezone
    from datetime import timedelta
    from task.models import Task
    
    # Buscar tarefas que vencem em 2 dias
    data_limite = timezone.now().date() + timedelta(days=2)
    
    tarefas_proximas = Task.objects.filter(
        cronograma_fim=data_limite,
        status__in=['nao_iniciada', 'em_andamento', 'pausada']
    )
    
    count = 0
    for task in tarefas_proximas:
        if task.responsavel:
            # Verificar se já existe notificação de prazo próximo
            existing = Notification.objects.filter(
                tipo='prazo_proximo',
                tarefa=task,
                destinatario=task.responsavel,
                created_at__date=timezone.now().date()
            ).exists()
            
            if not existing:
                Notification.criar_notificacao_prazo_proximo(task)
                count += 1
    
    return f"Criadas {count} notificações de prazos próximos"

