from django.shortcuts import render, get_object_or_404
from django.http import JsonResponse
from django.contrib.auth.decorators import login_required
from django.views.decorators.http import require_POST
from django.utils import timezone
from .models import Notification


@login_required
def notification_list(request):
    """
    Lista todas as notificações do usuário logado.
    """
    # Buscar notificações do usuário logado
    notifications = Notification.objects.filter(
        destinatario__user=request.user
    ).order_by('-created_at')
    
    context = {
        'notifications': notifications,
    }
    
    return render(request, 'notification/list.html', context)


@login_required
@require_POST
def mark_as_read(request, notification_id):
    """
    Marca uma notificação como lida.
    """
    try:
        notification = get_object_or_404(
            Notification, 
            id=notification_id,
            destinatario__user=request.user
        )
        
        notification.marcar_como_lida()
        
        return JsonResponse({
            'success': True,
            'message': 'Notificação marcada como lida'
        })
        
    except Exception as e:
        return JsonResponse({
            'success': False,
            'message': str(e)
        }, status=400)


@login_required
def notification_detail(request, notification_id):
    """
    Exibe detalhes de uma notificação específica.
    """
    notification = get_object_or_404(
        Notification, 
        id=notification_id,
        destinatario__user=request.user
    )
    
    # Marcar como lida se ainda não estiver
    if not notification.lida:
        notification.marcar_como_lida()
    
    context = {
        'notification': notification,
    }
    
    return render(request, 'notification/detail.html', context)


@login_required
def unread_count(request):
    """
    Retorna o número de notificações não lidas.
    """
    count = Notification.objects.filter(
        destinatario__user=request.user,
        lida=False
    ).count()
    
    return JsonResponse({'count': count})
