from django.contrib import admin
from django.utils.html import format_html
from django.urls import reverse
from django.utils.safestring import mark_safe
from .models import VisibilityPermission, WorkspaceAccess


@admin.register(VisibilityPermission)
class VisibilityPermissionAdmin(admin.ModelAdmin):
    """
    Admin para gerenciar permissões de visão hierárquicas.
    """
    list_display = [
        'permission_type', 'scope_type', 'scope_name', 'access_type', 
        'content_object_link', 'ativo', 'created_at'
    ]
    
    list_filter = [
        'permission_type', 'access_type', 'scope_type', 'ativo', 'created_at'
    ]
    
    search_fields = [
        'collaborator__nome', 'area__nome', 'content_type__model'
    ]
    
    readonly_fields = ['created_at', 'updated_at']
    
    fieldsets = (
        ('Informações Básicas', {
            'fields': ('permission_type', 'access_type', 'scope_type')
        }),
        ('Escopo da Permissão', {
            'fields': ('collaborator', 'area'),
            'description': 'Especifique um colaborador OU uma área inteira'
        }),
        ('Objeto de Acesso', {
            'fields': ('content_type', 'object_id'),
            'description': 'Selecione o tipo e ID do objeto para controlar acesso'
        }),
        ('Controle de Herança', {
            'fields': ('inherit_from_parent', 'override_parent'),
            'description': 'Controle como esta permissão interage com permissões do nível superior'
        }),
        ('Status', {
            'fields': ('ativo',)
        }),
        ('Timestamps', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    
    def scope_name(self, obj):
        """Exibe o nome do escopo (colaborador ou área)."""
        if obj.collaborator:
            return obj.collaborator.nome
        elif obj.area:
            return obj.area.nome
        return "N/A"
    scope_name.short_description = "Escopo"
    
    def content_object_link(self, obj):
        """Cria um link para o objeto de conteúdo."""
        if obj.content_object:
            try:
                url = reverse(f'admin:{obj.content_type.app_label}_{obj.content_type.model}_change', args=[obj.object_id])
                return format_html('<a href="{}">{}</a>', url, str(obj.content_object))
            except:
                return str(obj.content_object)
        return "N/A"
    content_object_link.short_description = "Objeto"
    
    def get_queryset(self, request):
        """Otimiza as consultas com select_related."""
        return super().get_queryset(request).select_related(
            'collaborator', 'area', 'content_type'
        )


@admin.register(WorkspaceAccess)
class WorkspaceAccessAdmin(admin.ModelAdmin):
    """
    Admin para gerenciar acesso simplificado a workspaces.
    """
    list_display = [
        'workspace', 'access_type', 'access_name', 'can_manage', 'ativo', 'created_at'
    ]
    
    list_filter = [
        'can_manage', 'ativo', 'created_at', 'workspace__cliente'
    ]
    
    search_fields = [
        'workspace__nome', 'collaborator__nome', 'area__nome'
    ]
    
    readonly_fields = ['created_at', 'updated_at']
    
    fieldsets = (
        ('Workspace', {
            'fields': ('workspace',)
        }),
        ('Acesso', {
            'fields': ('collaborator', 'area'),
            'description': 'Especifique um colaborador OU uma área inteira'
        }),
        ('Permissões', {
            'fields': ('can_manage',)
        }),
        ('Status', {
            'fields': ('ativo',)
        }),
        ('Timestamps', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    
    def access_type(self, obj):
        """Exibe o tipo de acesso (colaborador ou área)."""
        if obj.collaborator:
            return "Colaborador"
        elif obj.area:
            return "Área"
        return "N/A"
    access_type.short_description = "Tipo de Acesso"
    
    def access_name(self, obj):
        """Exibe o nome do colaborador ou área."""
        if obj.collaborator:
            return obj.collaborator.nome
        elif obj.area:
            return obj.area.nome
        return "N/A"
    access_name.short_description = "Nome"
    
    def get_queryset(self, request):
        """Otimiza as consultas com select_related."""
        return super().get_queryset(request).select_related(
            'workspace', 'collaborator', 'area'
        )
