"""
Exemplos de como integrar o sistema de permissões de visão nas views existentes.

Este arquivo demonstra como usar os mixins e funções do sistema de permissões
para controlar o acesso aos objetos hierárquicos.
"""

from django.shortcuts import render, get_object_or_404
from django.contrib.auth.decorators import login_required
from django.views.generic import ListView, DetailView, CreateView, UpdateView, DeleteView
from django.contrib.auth.mixins import LoginRequiredMixin
from django.contrib.contenttypes.models import ContentType

from workspace.models import Workspace
from portfolio.models import Portfolio
from project.models import Project
from board.models import Board

# Importar os mixins do sistema de permissões
from .mixins import (
    WorkspacePermissionMixin,
    PortfolioPermissionMixin,
    ProjectPermissionMixin,
    BoardPermissionMixin,
    ManagePermissionMixin,
    PermissionContextMixin
)
from .models import VisibilityPermission


# ============================================================================
# EXEMPLO 1: Views de Workspace com Controle de Permissões
# ============================================================================

class WorkspaceListView(WorkspacePermissionMixin, PermissionContextMixin, ListView):
    """
    Lista de workspaces com controle de permissões.
    O mixin WorkspacePermissionMixin automaticamente filtra apenas os workspaces
    que o usuário tem permissão para visualizar.
    """
    model = Workspace
    template_name = 'workspace/list.html'
    context_object_name = 'workspaces'
    paginate_by = 12

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        # Adiciona informações de permissões ao contexto
        context['can_manage_workspaces'] = self.request.user.is_superuser
        return context


class WorkspaceDetailView(WorkspacePermissionMixin, PermissionContextMixin, DetailView):
    """
    Detalhes do workspace com verificação de permissão.
    """
    model = Workspace
    template_name = 'workspace/detail.html'
    context_object_name = 'workspace'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        workspace = self.get_object()
        
        # Verifica se o usuário pode gerenciar este workspace
        context['can_manage'] = VisibilityPermission.has_access(
            self.request.user, workspace
        )
        
        # Adiciona link para gerenciar permissões se o usuário tiver acesso
        if context['can_manage']:
            context['permissions_url'] = f'/permissions/workspace/{workspace.id}/'
        
        return context


class WorkspaceCreateView(LoginRequiredMixin, CreateView):
    """
    Criação de workspace (apenas usuários logados).
    """
    model = Workspace
    template_name = 'workspace/form.html'
    fields = ['nome', 'cliente', 'descricao', 'cor']

    def form_valid(self, form):
        workspace = form.save()
        # Automaticamente dá permissão total ao criador
        from collaborator.models import Collaborator
        try:
            collaborator = Collaborator.objects.get(user=self.request.user)
            VisibilityPermission.objects.create(
                permission_type='workspace',
                scope_type='collaborator',
                collaborator=collaborator,
                content_type=ContentType.objects.get_for_model(workspace),
                object_id=workspace.id,
                access_type='allow',
                ativo=True
            )
        except Collaborator.DoesNotExist:
            pass
        
        return super().form_valid(form)


class WorkspaceUpdateView(ManagePermissionMixin, UpdateView):
    """
    Edição de workspace com verificação de permissão de gerenciamento.
    """
    model = Workspace
    template_name = 'workspace/form.html'
    fields = ['nome', 'cliente', 'descricao', 'cor']


class WorkspaceDeleteView(ManagePermissionMixin, DeleteView):
    """
    Exclusão de workspace com verificação de permissão de gerenciamento.
    """
    model = Workspace
    template_name = 'workspace/delete.html'
    success_url = '/workspaces/'


# ============================================================================
# EXEMPLO 2: Views de Portfolio com Controle de Permissões
# ============================================================================

class PortfolioListView(PortfolioPermissionMixin, PermissionContextMixin, ListView):
    """
    Lista de portfolios com controle de permissões.
    """
    model = Portfolio
    template_name = 'portfolio/list.html'
    context_object_name = 'portfolios'
    paginate_by = 12


class PortfolioDetailView(PortfolioPermissionMixin, PermissionContextMixin, DetailView):
    """
    Detalhes do portfolio com verificação de permissão.
    """
    model = Portfolio
    template_name = 'portfolio/detail.html'
    context_object_name = 'portfolio'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        portfolio = self.get_object()
        
        # Verifica se o usuário pode gerenciar este portfolio
        context['can_manage'] = VisibilityPermission.has_access(
            self.request.user, portfolio
        )
        
        return context


# ============================================================================
# EXEMPLO 3: Views de Projeto com Controle de Permissões
# ============================================================================

class ProjectListView(ProjectPermissionMixin, PermissionContextMixin, ListView):
    """
    Lista de projetos com controle de permissões.
    """
    model = Project
    template_name = 'project/list.html'
    context_object_name = 'projects'
    paginate_by = 12


class ProjectDetailView(ProjectPermissionMixin, PermissionContextMixin, DetailView):
    """
    Detalhes do projeto com verificação de permissão.
    """
    model = Project
    template_name = 'project/detail.html'
    context_object_name = 'project'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        project = self.get_object()
        
        # Verifica se o usuário pode gerenciar este projeto
        context['can_manage'] = VisibilityPermission.has_access(
            self.request.user, project
        )
        
        return context


# ============================================================================
# EXEMPLO 4: Views de Board com Controle de Permissões
# ============================================================================

class BoardListView(BoardPermissionMixin, PermissionContextMixin, ListView):
    """
    Lista de boards com controle de permissões.
    """
    model = Board
    template_name = 'board/list.html'
    context_object_name = 'boards'
    paginate_by = 12


class BoardDetailView(BoardPermissionMixin, PermissionContextMixin, DetailView):
    """
    Detalhes do board com verificação de permissão.
    """
    model = Board
    template_name = 'board/detail.html'
    context_object_name = 'board'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        board = self.get_object()
        
        # Verifica se o usuário pode gerenciar este board
        context['can_manage'] = VisibilityPermission.has_access(
            self.request.user, board
        )
        
        return context


# ============================================================================
# EXEMPLO 5: Views Function-Based com Controle de Permissões
# ============================================================================

@login_required
def workspace_list_with_permissions(request):
    """
    Exemplo de view function-based com controle de permissões.
    """
    # Obtém apenas os workspaces que o usuário tem permissão para ver
    workspaces = VisibilityPermission.get_accessible_objects(
        request.user, Workspace
    )
    
    # Aplica filtros adicionais
    search = request.GET.get('search', '')
    if search:
        workspaces = workspaces.filter(nome__icontains=search)
    
    context = {
        'workspaces': workspaces,
        'search': search,
    }
    
    return render(request, 'workspace/list.html', context)


@login_required
def workspace_detail_with_permissions(request, pk):
    """
    Exemplo de view function-based para detalhes com verificação de permissão.
    """
    workspace = get_object_or_404(Workspace, pk=pk)
    
    # Verifica se o usuário tem permissão para ver este workspace
    if not VisibilityPermission.has_access(request.user, workspace):
        from django.core.exceptions import PermissionDenied
        raise PermissionDenied("Você não tem permissão para acessar este workspace.")
    
    # Verifica se o usuário pode gerenciar este workspace
    can_manage = VisibilityPermission.has_access(request.user, workspace)
    
    context = {
        'workspace': workspace,
        'can_manage': can_manage,
    }
    
    return render(request, 'workspace/detail.html', context)


# ============================================================================
# EXEMPLO 6: API Views com Controle de Permissões
# ============================================================================

from rest_framework import generics
from rest_framework.permissions import IsAuthenticated
from .serializers import WorkspaceSerializer, PortfolioSerializer

class WorkspaceAPIList(generics.ListAPIView):
    """
    API para listar workspaces com controle de permissões.
    """
    serializer_class = WorkspaceSerializer
    permission_classes = [IsAuthenticated]
    
    def get_queryset(self):
        # Filtra apenas os workspaces que o usuário tem permissão para ver
        return VisibilityPermission.get_accessible_objects(
            self.request.user, Workspace
        )


class WorkspaceAPIDetail(generics.RetrieveAPIView):
    """
    API para detalhes do workspace com verificação de permissão.
    """
    serializer_class = WorkspaceSerializer
    permission_classes = [IsAuthenticated]
    
    def get_queryset(self):
        # Filtra apenas os workspaces que o usuário tem permissão para ver
        return VisibilityPermission.get_accessible_objects(
            self.request.user, Workspace
        )


# ============================================================================
# EXEMPLO 7: Templates com Controle de Permissões
# ============================================================================

"""
Exemplo de como usar no template:

{% if can_manage %}
    <a href="{% url 'workspace:edit' workspace.id %}" class="btn btn-primary">
        <i class="fas fa-edit"></i> Editar
    </a>
    <a href="{% url 'workspace:delete' workspace.id %}" class="btn btn-danger">
        <i class="fas fa-trash"></i> Excluir
    </a>
    <a href="{% url 'permissions:workspace_permissions' workspace.id %}" class="btn btn-info">
        <i class="fas fa-shield-alt"></i> Gerenciar Permissões
    </a>
{% endif %}

{% if user_permissions.workspaces %}
    <div class="alert alert-info">
        <i class="fas fa-info-circle"></i>
        Você tem permissões específicas para este workspace.
    </div>
{% endif %}
"""


# ============================================================================
# EXEMPLO 8: Middleware Personalizado
# ============================================================================

"""
Exemplo de middleware para verificar permissões automaticamente:

class PermissionMiddleware:
    def __init__(self, get_response):
        self.get_response = get_response

    def __call__(self, request):
        response = self.get_response(request)
        return response

    def process_view(self, request, view_func, view_args, view_kwargs):
        # Verifica permissões para views específicas
        if hasattr(view_func, 'model') and view_func.model in [Workspace, Portfolio, Project, Board]:
            # Lógica de verificação de permissões
            pass
        return None
"""


# ============================================================================
# EXEMPLO 9: Comandos de Gerenciamento
# ============================================================================

"""
Exemplo de comando para migrar permissões existentes:

from django.core.management.base import BaseCommand
from permissions.models import VisibilityPermission

class Command(BaseCommand):
    help = 'Migra permissões existentes para o novo sistema'

    def handle(self, *args, **options):
        # Lógica para migrar permissões existentes
        pass
"""


# ============================================================================
# EXEMPLO 10: Testes
# ============================================================================

"""
Exemplo de testes para o sistema de permissões:

from django.test import TestCase
from django.contrib.auth.models import User
from permissions.models import VisibilityPermission
from workspace.models import Workspace

class PermissionTestCase(TestCase):
    def setUp(self):
        self.user = User.objects.create_user(username='testuser', password='testpass')
        self.workspace = Workspace.objects.create(nome='Test Workspace')
        
    def test_workspace_access(self):
        # Testa acesso sem permissão
        self.assertFalse(VisibilityPermission.has_access(self.user, self.workspace))
        
        # Cria permissão
        permission = VisibilityPermission.objects.create(
            permission_type='workspace',
            scope_type='collaborator',
            collaborator=self.user.collaborator,
            content_type=ContentType.objects.get_for_model(self.workspace),
            object_id=self.workspace.id,
            access_type='allow'
        )
        
        # Testa acesso com permissão
        self.assertTrue(VisibilityPermission.has_access(self.user, self.workspace))
""" 