from django.core.management.base import BaseCommand
from django.contrib.auth.models import User
from django.contrib.contenttypes.models import ContentType
from permissions.models import VisibilityPermission
from workspace.models import Workspace
from collaborator.models import Collaborator


class Command(BaseCommand):
    help = 'Verifica as permissões de um usuário específico'

    def add_arguments(self, parser):
        parser.add_argument(
            '--username',
            type=str,
            required=True,
            help='Nome do usuário para verificar permissões',
        )

    def handle(self, *args, **options):
        username = options['username']

        # Verifica se o usuário existe
        try:
            user = User.objects.get(username=username)
        except User.DoesNotExist:
            self.stdout.write(
                self.style.ERROR(f'Usuário "{username}" não encontrado.')
            )
            return

        # Verifica se o colaborador existe
        try:
            collaborator = Collaborator.objects.get(user=user)
        except Collaborator.DoesNotExist:
            self.stdout.write(
                self.style.ERROR(f'Colaborador para o usuário "{username}" não encontrado.')
            )
            return

        self.stdout.write(
            self.style.SUCCESS(f'\n=== PERMISSÕES DO USUÁRIO: {username} ===\n')
        )

        # Verifica se é superuser
        if user.is_superuser:
            self.stdout.write(
                self.style.WARNING('⚠️  Este usuário é SUPERUSER - tem acesso a tudo!')
            )
            return

        # Busca todas as permissões do colaborador
        permissions = VisibilityPermission.objects.filter(
            collaborator=collaborator,
            ativo=True
        ).select_related('content_type')

        if not permissions.exists():
            self.stdout.write(
                self.style.WARNING('❌ Nenhuma permissão encontrada para este usuário.')
            )
            return

        self.stdout.write(
            self.style.SUCCESS(f'📋 Total de permissões: {permissions.count()}\n')
        )

        # Agrupa por tipo de permissão
        workspace_permissions = permissions.filter(permission_type='workspace')
        portfolio_permissions = permissions.filter(permission_type='portfolio')
        project_permissions = permissions.filter(permission_type='project')
        board_permissions = permissions.filter(permission_type='board')

        # Mostra permissões de workspace
        if workspace_permissions.exists():
            self.stdout.write(
                self.style.SUCCESS('🏢 PERMISSÕES DE WORKSPACE:')
            )
            for perm in workspace_permissions:
                try:
                    workspace = Workspace.objects.get(id=perm.object_id)
                    access_type = "✅ PERMITIR" if perm.access_type == 'allow' else "❌ NEGAR"
                    self.stdout.write(
                        f'  • {workspace.nome} ({workspace.cliente.nome}) - {access_type}'
                    )
                except Workspace.DoesNotExist:
                    self.stdout.write(
                        f'  • Workspace ID {perm.object_id} (não encontrado) - {perm.access_type}'
                    )

        # Mostra permissões de portfolio
        if portfolio_permissions.exists():
            self.stdout.write(
                self.style.SUCCESS('\n📁 PERMISSÕES DE PORTFOLIO:')
            )
            for perm in portfolio_permissions:
                access_type = "✅ PERMITIR" if perm.access_type == 'allow' else "❌ NEGAR"
                self.stdout.write(
                    f'  • Portfolio ID {perm.object_id} - {access_type}'
                )

        # Mostra permissões de projeto
        if project_permissions.exists():
            self.stdout.write(
                self.style.SUCCESS('\n📋 PERMISSÕES DE PROJETO:')
            )
            for perm in project_permissions:
                access_type = "✅ PERMITIR" if perm.access_type == 'allow' else "❌ NEGAR"
                self.stdout.write(
                    f'  • Projeto ID {perm.object_id} - {access_type}'
                )

        # Mostra permissões de board
        if board_permissions.exists():
            self.stdout.write(
                self.style.SUCCESS('\n📊 PERMISSÕES DE BOARD:')
            )
            for perm in board_permissions:
                access_type = "✅ PERMITIR" if perm.access_type == 'allow' else "❌ NEGAR"
                self.stdout.write(
                    f'  • Board ID {perm.object_id} - {access_type}'
                )

        # Testa o método get_accessible_objects
        self.stdout.write(
            self.style.SUCCESS('\n🔍 TESTE DE ACESSO A WORKSPACES:')
        )
        
        # Teste de acesso a workspaces (permissões diretas)
        accessible_workspaces = VisibilityPermission.get_accessible_objects(user, Workspace)
        self.stdout.write(
            f'Workspaces acessíveis (permissões diretas): {accessible_workspaces.count()}'
        )
        
        for workspace in accessible_workspaces:
            self.stdout.write(
                f'  • {workspace.nome} ({workspace.cliente.nome})'
            )

        # Lista todos os workspaces para comparação
        all_workspaces = Workspace.objects.all()
        self.stdout.write(
            self.style.SUCCESS(f'\n📋 TODOS OS WORKSPACES ({all_workspaces.count()}):')
        )
        
        for workspace in all_workspaces:
            has_access = VisibilityPermission.has_access(user, workspace)
            status = "✅ ACESSO" if has_access else "❌ SEM ACESSO"
            self.stdout.write(
                f'  • {workspace.nome} ({workspace.cliente.nome}) - {status}'
            ) 