from django.core.management.base import BaseCommand
from django.contrib.auth.models import User
from django.contrib.contenttypes.models import ContentType
from permissions.models import VisibilityPermission
from workspace.models import Workspace
from portfolio.models import Portfolio
from project.models import Project
from board.models import Board
from collaborator.models import Collaborator, Area


class Command(BaseCommand):
    help = 'Cria permissões de exemplo para testar o sistema de controle de visão'

    def add_arguments(self, parser):
        parser.add_argument(
            '--username',
            type=str,
            help='Nome do usuário para criar permissões',
        )
        parser.add_argument(
            '--workspace-id',
            type=int,
            help='ID do workspace para criar permissões',
        )

    def handle(self, *args, **options):
        username = options['username']
        workspace_id = options['workspace_id']

        # Verifica se o usuário existe
        try:
            user = User.objects.get(username=username)
        except User.DoesNotExist:
            self.stdout.write(
                self.style.ERROR(f'Usuário "{username}" não encontrado.')
            )
            return

        # Verifica se o colaborador existe
        try:
            collaborator = Collaborator.objects.get(user=user)
        except Collaborator.DoesNotExist:
            self.stdout.write(
                self.style.ERROR(f'Colaborador para o usuário "{username}" não encontrado.')
            )
            return

        # Se workspace_id foi especificado, usa apenas ele
        if workspace_id:
            try:
                workspace = Workspace.objects.get(id=workspace_id)
                workspaces = [workspace]
            except Workspace.DoesNotExist:
                self.stdout.write(
                    self.style.ERROR(f'Workspace com ID {workspace_id} não encontrado.')
                )
                return
        else:
            # Usa todos os workspaces
            workspaces = Workspace.objects.all()

        if not workspaces:
            self.stdout.write(
                self.style.WARNING('Nenhum workspace encontrado.')
            )
            return

        # Cria permissões para cada workspace
        for workspace in workspaces:
            # Permissão para workspace
            permission, created = VisibilityPermission.objects.get_or_create(
                permission_type='workspace',
                scope_type='collaborator',
                collaborator=collaborator,
                content_type=ContentType.objects.get_for_model(workspace),
                object_id=workspace.id,
                defaults={
                    'access_type': 'allow',
                    'ativo': True
                }
            )

            if created:
                self.stdout.write(
                    self.style.SUCCESS(f'Permissão criada para workspace "{workspace.nome}"')
                )
            else:
                self.stdout.write(
                    self.style.WARNING(f'Permissão já existe para workspace "{workspace.nome}"')
                )

            # Cria permissões para portfolios do workspace
            for portfolio in workspace.portfolios.all():
                portfolio_permission, created = VisibilityPermission.objects.get_or_create(
                    permission_type='portfolio',
                    scope_type='collaborator',
                    collaborator=collaborator,
                    content_type=ContentType.objects.get_for_model(portfolio),
                    object_id=portfolio.id,
                    defaults={
                        'access_type': 'allow',
                        'ativo': True
                    }
                )

                if created:
                    self.stdout.write(
                        self.style.SUCCESS(f'  - Permissão criada para portfolio "{portfolio.nome}"')
                    )

                # Cria permissões para projetos do portfolio
                for project in portfolio.projects.all():
                    project_permission, created = VisibilityPermission.objects.get_or_create(
                        permission_type='project',
                        scope_type='collaborator',
                        collaborator=collaborator,
                        content_type=ContentType.objects.get_for_model(project),
                        object_id=project.id,
                        defaults={
                            'access_type': 'allow',
                            'ativo': True
                        }
                    )

                    if created:
                        self.stdout.write(
                            self.style.SUCCESS(f'    - Permissão criada para projeto "{project.nome}"')
                        )

                    # Cria permissões para boards do projeto
                    for board in project.boards.all():
                        board_permission, created = VisibilityPermission.objects.get_or_create(
                            permission_type='board',
                            scope_type='collaborator',
                            collaborator=collaborator,
                            content_type=ContentType.objects.get_for_model(board),
                            object_id=board.id,
                            defaults={
                                'access_type': 'allow',
                                'ativo': True
                            }
                        )

                        if created:
                            self.stdout.write(
                                self.style.SUCCESS(f'      - Permissão criada para board "{board.nome}"')
                            )

        self.stdout.write(
            self.style.SUCCESS(f'\nPermissões criadas com sucesso para o usuário "{username}"!')
        )

        # Mostra estatísticas
        total_permissions = VisibilityPermission.objects.filter(
            scope_type='collaborator',
            collaborator=collaborator,
            ativo=True
        ).count()

        self.stdout.write(
            self.style.SUCCESS(f'Total de permissões ativas para {username}: {total_permissions}')
        ) 