from django.core.management.base import BaseCommand
from django.contrib.auth.models import User
from django.contrib.auth import authenticate
from permissions.models import VisibilityPermission
from workspace.models import Workspace


class Command(BaseCommand):
    help = 'Debuga problema de permissão para remoção'

    def add_arguments(self, parser):
        parser.add_argument(
            '--username',
            type=str,
            default='admin',
            help='Nome do usuário para testar',
        )
        parser.add_argument(
            '--password',
            type=str,
            default='admin123',
            help='Senha do usuário',
        )

    def handle(self, *args, **options):
        username = options['username']
        password = options['password']

        self.stdout.write(
            self.style.SUCCESS(f'\n=== DEBUG PERMISSÃO PARA: {username} ===\n')
        )

        # Verifica se o usuário existe
        try:
            user = User.objects.get(username=username)
        except User.DoesNotExist:
            self.stdout.write(
                self.style.ERROR(f'❌ Usuário "{username}" não encontrado.')
            )
            return

        self.stdout.write(f'✅ Usuário encontrado: {user.username}')
        self.stdout.write(f'   - is_superuser: {user.is_superuser}')
        self.stdout.write(f'   - is_staff: {user.is_staff}')
        self.stdout.write(f'   - is_active: {user.is_active}')
        self.stdout.write(f'   - is_authenticated: {user.is_authenticated}')

        # Testa autenticação
        authenticated_user = authenticate(username=username, password=password)
        if authenticated_user:
            self.stdout.write(f'✅ Autenticação bem-sucedida')
        else:
            self.stdout.write(f'❌ Falha na autenticação')

        # Verifica permissões específicas
        self.stdout.write(
            self.style.SUCCESS('\n🔍 VERIFICANDO PERMISSÕES ESPECÍFICAS:')
        )

        # Testa se pode acessar workspace ID 1
        try:
            workspace = Workspace.objects.get(id=1)
            self.stdout.write(f'✅ Workspace ID 1 encontrado: {workspace.nome}')
        except Workspace.DoesNotExist:
            self.stdout.write(f'❌ Workspace ID 1 não encontrado')
            return

        # Testa se pode acessar permissão ID 8
        try:
            permission = VisibilityPermission.objects.get(id=8)
            self.stdout.write(f'✅ Permissão ID 8 encontrada')
            self.stdout.write(f'   - Tipo: {permission.permission_type}')
            self.stdout.write(f'   - Escopo: {permission.scope_type}')
            self.stdout.write(f'   - Ativo: {permission.ativo}')
            if permission.collaborator:
                self.stdout.write(f'   - Colaborador: {permission.collaborator.nome}')
            if permission.area:
                self.stdout.write(f'   - Área: {permission.area.nome}')
        except VisibilityPermission.DoesNotExist:
            self.stdout.write(f'❌ Permissão ID 8 não encontrada')
            return

        # Testa se a permissão pertence ao workspace
        if permission.content_type.model == 'workspace' and permission.object_id == 1:
            self.stdout.write(f'✅ Permissão pertence ao workspace correto')
        else:
            self.stdout.write(f'❌ Permissão NÃO pertence ao workspace correto')
            self.stdout.write(f'   - Content type: {permission.content_type.model}')
            self.stdout.write(f'   - Object ID: {permission.object_id}')

        # Testa se o usuário tem acesso ao workspace
        has_access = VisibilityPermission.has_access(user, workspace)
        self.stdout.write(f'   - Usuário tem acesso ao workspace: {has_access}')

        # Simula a verificação da view
        self.stdout.write(
            self.style.SUCCESS('\n🔍 SIMULANDO VERIFICAÇÃO DA VIEW:')
        )
        
        if user.is_superuser:
            self.stdout.write(f'✅ Usuário é superuser - deve ter acesso')
        else:
            self.stdout.write(f'❌ Usuário NÃO é superuser - acesso negado')

        # Lista todas as permissões do workspace
        self.stdout.write(
            self.style.SUCCESS('\n📋 TODAS AS PERMISSÕES DO WORKSPACE:')
        )
        
        workspace_permissions = VisibilityPermission.objects.filter(
            content_type__model='workspace',
            object_id=1,
            ativo=True
        )
        
        for perm in workspace_permissions:
            scope_name = perm.collaborator.nome if perm.collaborator else perm.area.nome
            self.stdout.write(f'  • ID {perm.id}: {scope_name} - {perm.access_type}') 