from django.core.management.base import BaseCommand
from django.contrib.auth.models import User
from collaborator.models import Collaborator, Area


class Command(BaseCommand):
    help = 'Configura um colaborador para o usuário admin'

    def handle(self, *args, **options):
        # Verifica se o usuário admin existe
        try:
            admin_user = User.objects.get(username='admin')
        except User.DoesNotExist:
            self.stdout.write(
                self.style.ERROR('Usuário admin não encontrado. Crie primeiro com: python manage.py createsuperuser')
            )
            return

        # Verifica se já existe um colaborador para o admin
        if Collaborator.objects.filter(user=admin_user).exists():
            self.stdout.write(
                self.style.WARNING('Colaborador para o usuário admin já existe.')
            )
            return

        # Cria uma área padrão se não existir
        area, created = Area.objects.get_or_create(
            nome='Administração',
            defaults={
                'descricao': 'Área administrativa do sistema',
                'ativo': True
            }
        )

        if created:
            self.stdout.write(
                self.style.SUCCESS(f'Área "{area.nome}" criada.')
            )

        # Cria o colaborador para o admin
        collaborator = Collaborator.objects.create(
            user=admin_user,
            nome=admin_user.get_full_name() or admin_user.username,
            email=admin_user.email or 'admin@example.com',
            area=area,
            ativo=True
        )

        self.stdout.write(
            self.style.SUCCESS(f'Colaborador criado para o usuário admin: {collaborator.nome}')
        ) 