from django.core.management.base import BaseCommand
from django.contrib.auth.models import User
from workspace.models import Workspace
from permissions.models import VisibilityPermission


class Command(BaseCommand):
    help = 'Testa o acesso de usuários a workspaces'

    def add_arguments(self, parser):
        parser.add_argument('--username', type=str, help='Nome do usuário para testar')

    def handle(self, *args, **options):
        username = options['username']
        
        try:
            user = User.objects.get(username=username)
        except User.DoesNotExist:
            self.stdout.write(
                self.style.ERROR(f'❌ Usuário "{username}" não encontrado!')
            )
            return
        
        self.stdout.write(f'=== TESTE DE ACESSO: {username} ===')
        
        # Verifica se é superuser
        if user.is_superuser:
            self.stdout.write(
                self.style.SUCCESS('⚠️  Este usuário é SUPERUSER - tem acesso a tudo!')
            )
        
        # Lista todos os workspaces
        all_workspaces = Workspace.objects.all()
        self.stdout.write(f'\n📋 TODOS OS WORKSPACES ({all_workspaces.count()}):')
        
        for workspace in all_workspaces:
            has_access = VisibilityPermission.has_access(user, workspace)
            status = '✅ ACESSO' if has_access else '❌ SEM ACESSO'
            self.stdout.write(
                f'  • {workspace.nome} ({workspace.cliente.nome}) - {status}'
            )
        
        # Lista workspaces acessíveis
        accessible_workspaces = VisibilityPermission.get_accessible_objects(user, Workspace)
        self.stdout.write(f'\n🎯 WORKSPACES ACESSÍVEIS: {accessible_workspaces.count()}')
        
        for workspace in accessible_workspaces:
            self.stdout.write(
                f'  • {workspace.nome} ({workspace.cliente.nome})'
            ) 