from django.core.management.base import BaseCommand
from django.contrib.auth.models import User
from workspace.models import Workspace
from portfolio.models import Portfolio
from project.models import Project
from board.models import Board
from permissions.models import VisibilityPermission


class Command(BaseCommand):
    help = 'Testa o acesso de usuários a todos os níveis'

    def add_arguments(self, parser):
        parser.add_argument('--username', type=str, help='Nome do usuário para testar')

    def handle(self, *args, **options):
        username = options['username']
        
        try:
            user = User.objects.get(username=username)
        except User.DoesNotExist:
            self.stdout.write(
                self.style.ERROR(f'❌ Usuário "{username}" não encontrado!')
            )
            return
        
        self.stdout.write(f'=== TESTE COMPLETO DE ACESSO: {username} ===')
        
        # Verifica se é superuser
        if user.is_superuser:
            self.stdout.write(
                self.style.SUCCESS('⚠️  Este usuário é SUPERUSER - tem acesso a tudo!')
            )
        
        # Testa cada nível
        levels = [
            ('Workspace', Workspace),
            ('Portfolio', Portfolio),
            ('Project', Project),
            ('Board', Board),
        ]
        
        for level_name, model_class in levels:
            self.stdout.write(f'\n📋 {level_name.upper()}S:')
            
            # Lista todos os objetos
            all_objects = model_class.objects.all()
            accessible_objects = VisibilityPermission.get_accessible_objects(user, model_class)
            
            for obj in all_objects:
                has_access = VisibilityPermission.has_access(user, obj)
                status = '✅ ACESSO' if has_access else '❌ SEM ACESSO'
                
                # Nome do objeto baseado no tipo
                if level_name == 'Workspace':
                    obj_name = f"{obj.nome} ({obj.cliente.nome})"
                elif level_name == 'Portfolio':
                    obj_name = f"{obj.nome} ({obj.workspace.nome})"
                elif level_name == 'Project':
                    obj_name = f"{obj.nome} ({obj.portfolio.nome})"
                elif level_name == 'Board':
                    obj_name = f"{obj.nome} ({obj.project.nome})"
                else:
                    obj_name = str(obj)
                
                self.stdout.write(f'  • {obj_name} - {status}')
            
            self.stdout.write(f'🎯 {level_name}s acessíveis: {accessible_objects.count()}') 