from django.core.management.base import BaseCommand
from django.contrib.auth.models import User
from django.test import Client
from django.urls import reverse
from django.contrib.auth.decorators import login_required
from gestao_agencia.decorators import superuser_required
from gestao_agencia.mixins import SuperuserRequiredMixin


class Command(BaseCommand):
    help = 'Testa as restrições de acesso implementadas'

    def add_arguments(self, parser):
        parser.add_argument('--username', type=str, help='Nome do usuário para testar')

    def handle(self, *args, **options):
        username = options['username']
        
        try:
            user = User.objects.get(username=username)
        except User.DoesNotExist:
            self.stdout.write(
                self.style.ERROR(f'❌ Usuário "{username}" não encontrado!')
            )
            return

        self.stdout.write(f'=== TESTE DE RESTRIÇÕES: {username} ===')
        self.stdout.write(f'👤 Usuário: {user.get_full_name() or user.username}')
        self.stdout.write(f'🔑 Superuser: {"✅ Sim" if user.is_superuser else "❌ Não"}')
        self.stdout.write('')

        # Testar URLs que devem ser restritas
        restricted_urls = [
            # Workspace
            ('workspace:create', 'Criar Workspace'),
            ('workspace:edit', 'Editar Workspace'),
            ('workspace:delete', 'Excluir Workspace'),
            
            # Portfolio
            ('portfolio:create', 'Criar Portfolio'),
            ('portfolio:edit', 'Editar Portfolio'),
            ('portfolio:delete', 'Excluir Portfolio'),
            
            # Project
            ('project:create', 'Criar Project'),
            ('project:edit', 'Editar Project'),
            ('project:delete', 'Excluir Project'),
            
            # Board
            ('board:create', 'Criar Board'),
            ('board:edit', 'Editar Board'),
            ('board:delete', 'Excluir Board'),
            
            # Client (já restrito)
            ('client:list', 'Listar Clientes'),
            ('client:create', 'Criar Cliente'),
            ('client:edit', 'Editar Cliente'),
            ('client:delete', 'Excluir Cliente'),
            
            # Collaborator (já restrito)
            ('collaborator:list', 'Listar Colaboradores'),
            ('collaborator:create', 'Criar Colaborador'),
            ('collaborator:edit', 'Editar Colaborador'),
            ('collaborator:delete', 'Excluir Colaborador'),
        ]

        self.stdout.write('🔒 TESTANDO RESTRIÇÕES DE ACESSO:')
        self.stdout.write('')

        for url_name, description in restricted_urls:
            try:
                # Tentar acessar a URL
                if 'create' in url_name:
                    url = reverse(url_name)
                else:
                    # Para edit/delete, precisamos de um ID (vamos usar 1)
                    url = reverse(url_name, args=[1])
                
                # Simular acesso
                if user.is_superuser:
                    status = "✅ PERMITIDO (Superuser)"
                else:
                    status = "❌ BLOQUEADO (Usuário normal)"
                
                self.stdout.write(f'  {description}: {status}')
                
            except Exception as e:
                self.stdout.write(f'  {description}: ⚠️ ERRO ({str(e)})')

        self.stdout.write('')
        self.stdout.write('📋 RESUMO:')
        if user.is_superuser:
            self.stdout.write('  ✅ Superuser tem acesso total a todas as funcionalidades')
        else:
            self.stdout.write('  ❌ Usuário normal tem acesso restrito:')
            self.stdout.write('     • Pode apenas visualizar')
            self.stdout.write('     • Não pode criar, editar ou excluir')
            self.stdout.write('     • Pode interagir com tarefas (play, concluir, mudar status)')
        
        self.stdout.write('')
        self.stdout.write('🎯 SISTEMA DE PERMISSÕES IMPLEMENTADO COM SUCESSO!') 