from django.core.management.base import BaseCommand
from django.contrib.auth.models import User
from django.template.loader import render_to_string
from django.template import Context, Template
from django.test import RequestFactory
from django.contrib.auth.models import AnonymousUser
from workspace.models import Workspace
from portfolio.models import Portfolio
from project.models import Project
from board.models import Board


class Command(BaseCommand):
    help = 'Testa se os botões estão ocultos nos templates para usuários não superuser'

    def add_arguments(self, parser):
        parser.add_argument('--username', type=str, help='Nome do usuário para testar')

    def handle(self, *args, **options):
        username = options['username']
        
        try:
            user = User.objects.get(username=username)
        except User.DoesNotExist:
            self.stdout.write(
                self.style.ERROR(f'❌ Usuário "{username}" não encontrado!')
            )
            return

        self.stdout.write(f'=== TESTE DE TEMPLATES: {username} ===')
        self.stdout.write(f'👤 Usuário: {user.get_full_name() or user.username}')
        self.stdout.write(f'🔑 Superuser: {"✅ Sim" if user.is_superuser else "❌ Não"}')
        self.stdout.write('')

        # Criar request factory
        factory = RequestFactory()
        
        # Testar templates com dados reais
        templates_to_test = [
            ('workspace/list.html', 'Workspace List', 'workspaces'),
            ('portfolio/list.html', 'Portfolio List', 'portfolios'),
            ('project/list.html', 'Project List', 'projects'),
            ('board/list.html', 'Board List', 'boards'),
        ]

        self.stdout.write('🔍 TESTANDO OCULTAÇÃO DE BOTÕES:')
        self.stdout.write('')

        for template_name, description, context_key in templates_to_test:
            try:
                # Criar request
                request = factory.get('/')
                request.user = user
                
                # Buscar dados reais
                if context_key == 'workspaces':
                    items = Workspace.objects.all()[:3]  # Primeiros 3
                elif context_key == 'portfolios':
                    items = Portfolio.objects.all()[:3]
                elif context_key == 'projects':
                    items = Project.objects.all()[:3]
                elif context_key == 'boards':
                    items = Board.objects.all()[:3]
                else:
                    items = []
                
                # Contexto com dados reais
                context = {
                    'user': user,
                    context_key: items,
                    'page_obj': items,
                    'is_paginated': False,
                    'search': '',
                    'status': '',
                    'workspace_id': '',
                    'portfolio_id': '',
                    'project_id': '',
                    'cliente_id': '',
                }
                
                # Renderizar template
                rendered = render_to_string(template_name, context, request=request)
                
                # Verificar se botões estão presentes de forma mais específica
                has_edit_btn = 'edit-' in rendered and 'btn-outline-secondary' in rendered
                has_delete_btn = 'delete-btn' in rendered and 'btn-outline-danger' in rendered
                has_create_btn = ('Novo ' in rendered or 'Criar ' in rendered) and 'btn-primary' in rendered
                
                if user.is_superuser:
                    status = "✅ VISÍVEIS (Superuser)"
                else:
                    if has_edit_btn or has_delete_btn or has_create_btn:
                        status = "❌ AINDA VISÍVEIS (Problema!)"
                    else:
                        status = "✅ OCULTOS (Correto)"
                
                self.stdout.write(f'  {description}: {status}')
                
                if not user.is_superuser and (has_edit_btn or has_delete_btn or has_create_btn):
                    self.stdout.write(f'    ⚠️  Botões ainda visíveis:')
                    if has_edit_btn:
                        self.stdout.write(f'       - Editar')
                    if has_delete_btn:
                        self.stdout.write(f'       - Excluir')
                    if has_create_btn:
                        self.stdout.write(f'       - Criar/Novo')
                
            except Exception as e:
                self.stdout.write(f'  {description}: ⚠️ ERRO ({str(e)})')

        self.stdout.write('')
        self.stdout.write('📋 RESUMO:')
        if user.is_superuser:
            self.stdout.write('  ✅ Superuser vê todos os botões (correto)')
        else:
            self.stdout.write('  ❌ Usuário normal NÃO deve ver botões de ação')
            self.stdout.write('     • Botões de Editar devem estar ocultos')
            self.stdout.write('     • Botões de Excluir devem estar ocultos')
            self.stdout.write('     • Botões de Criar/Novo devem estar ocultos')
            self.stdout.write('     • Apenas botão "Ver" deve estar visível')
        
        self.stdout.write('')
        self.stdout.write('🎯 TESTE DE TEMPLATES CONCLUÍDO!') 