from django.contrib.auth.mixins import LoginRequiredMixin, UserPassesTestMixin
from django.core.exceptions import PermissionDenied
from django.shortcuts import redirect
from django.contrib import messages
from django.urls import reverse
from .models import VisibilityPermission


class VisibilityPermissionMixin(LoginRequiredMixin):
    """
    Mixin para controlar acesso baseado em permissões de visão.
    """
    
    def dispatch(self, request, *args, **kwargs):
        """Verifica permissões antes de processar a requisição."""
        if not request.user.is_authenticated:
            return self.handle_no_permission()
        
        # Verifica se o usuário tem acesso ao objeto
        if hasattr(self, 'get_object'):
            try:
                obj = self.get_object()
                if not VisibilityPermission.has_access(request.user, obj):
                    messages.error(request, "Você não tem permissão para acessar este item.")
                    return redirect(self.get_no_permission_url())
            except Exception:
                # Se não conseguir obter o objeto, continua (pode ser uma list view)
                pass
        
        return super().dispatch(request, *args, **kwargs)
    
    def get_no_permission_url(self):
        """URL para redirecionar quando não há permissão."""
        return reverse('home')


class WorkspacePermissionMixin(VisibilityPermissionMixin):
    """
    Mixin específico para controle de acesso a workspaces.
    """
    
    def get_queryset(self):
        """Filtra o queryset baseado nas permissões do usuário."""
        queryset = super().get_queryset()
        return VisibilityPermission.get_accessible_objects(
            self.request.user, 
            queryset.model
        )


class PortfolioPermissionMixin(VisibilityPermissionMixin):
    """
    Mixin específico para controle de acesso a portfolios.
    """
    
    def get_queryset(self):
        """Filtra o queryset baseado nas permissões do usuário."""
        queryset = super().get_queryset()
        return VisibilityPermission.get_accessible_objects(
            self.request.user, 
            queryset.model
        )


class ProjectPermissionMixin(VisibilityPermissionMixin):
    """
    Mixin específico para controle de acesso a projetos.
    """
    
    def get_queryset(self):
        """Filtra o queryset baseado nas permissões do usuário."""
        queryset = super().get_queryset()
        return VisibilityPermission.get_accessible_objects(
            self.request.user, 
            queryset.model
        )


class BoardPermissionMixin(VisibilityPermissionMixin):
    """
    Mixin específico para controle de acesso a boards.
    """
    
    def get_queryset(self):
        """Filtra o queryset baseado nas permissões do usuário."""
        queryset = super().get_queryset()
        return VisibilityPermission.get_accessible_objects(
            self.request.user, 
            queryset.model
        )


class ManagePermissionMixin(UserPassesTestMixin):
    """
    Mixin para verificar se o usuário pode gerenciar um objeto.
    """
    
    def test_func(self):
        """Testa se o usuário pode gerenciar o objeto."""
        if not self.request.user.is_authenticated:
            return False
        
        # Verifica se o usuário é superusuário
        if self.request.user.is_superuser:
            return True
        
        # Verifica se o usuário tem permissões de gerenciamento
        if hasattr(self, 'get_object'):
            try:
                obj = self.get_object()
                # Aqui você pode implementar lógica específica para verificar
                # se o usuário pode gerenciar o objeto
                return self.can_manage_object(obj)
            except Exception:
                return False
        
        return False
    
    def can_manage_object(self, obj):
        """
        Verifica se o usuário pode gerenciar o objeto específico.
        Pode ser sobrescrito em subclasses para lógica específica.
        """
        # Por padrão, verifica se tem acesso de visualização
        return VisibilityPermission.has_access(self.request.user, obj)
    
    def handle_no_permission(self):
        """Trata quando o usuário não tem permissão."""
        messages.error(self.request, "Você não tem permissão para gerenciar este item.")
        return redirect(self.get_no_permission_url())
    
    def get_no_permission_url(self):
        """URL para redirecionar quando não há permissão."""
        return reverse('home')


class PermissionContextMixin:
    """
    Mixin para adicionar contexto de permissões às views.
    """
    
    def get_context_data(self, **kwargs):
        """Adiciona informações de permissões ao contexto."""
        context = super().get_context_data(**kwargs)
        
        if self.request.user.is_authenticated:
            context['user_permissions'] = self.get_user_permissions()
            context['can_manage'] = self.can_manage_current_object()
        
        return context
    
    def get_user_permissions(self):
        """
        Retorna as permissões do usuário atual.
        Pode ser sobrescrito em subclasses para lógica específica.
        """
        from .models import VisibilityPermission
        
        if not self.request.user.is_authenticated:
            return {}
        
        # Obtém permissões do usuário
        permissions = {}
        
        # Permissões de workspace
        workspace_permissions = VisibilityPermission.objects.filter(
            scope_type='collaborator',
            collaborator__user=self.request.user,
            permission_type='workspace',
            ativo=True
        )
        permissions['workspaces'] = workspace_permissions
        
        # Permissões de portfolio
        portfolio_permissions = VisibilityPermission.objects.filter(
            scope_type='collaborator',
            collaborator__user=self.request.user,
            permission_type='portfolio',
            ativo=True
        )
        permissions['portfolios'] = portfolio_permissions
        
        # Permissões de projeto
        project_permissions = VisibilityPermission.objects.filter(
            scope_type='collaborator',
            collaborator__user=self.request.user,
            permission_type='project',
            ativo=True
        )
        permissions['projects'] = project_permissions
        
        # Permissões de board
        board_permissions = VisibilityPermission.objects.filter(
            scope_type='collaborator',
            collaborator__user=self.request.user,
            permission_type='board',
            ativo=True
        )
        permissions['boards'] = board_permissions
        
        return permissions
    
    def can_manage_current_object(self):
        """
        Verifica se o usuário pode gerenciar o objeto atual.
        Pode ser sobrescrito em subclasses para lógica específica.
        """
        if not self.request.user.is_authenticated:
            return False
        
        if self.request.user.is_superuser:
            return True
        
        if hasattr(self, 'get_object'):
            try:
                obj = self.get_object()
                return self.can_manage_object(obj)
            except Exception:
                return False
        
        return False
    
    def can_manage_object(self, obj):
        """
        Verifica se o usuário pode gerenciar o objeto específico.
        Pode ser sobrescrito em subclasses para lógica específica.
        """
        from .models import VisibilityPermission
        return VisibilityPermission.has_access(self.request.user, obj) 