from django.db import models
from django.contrib.auth.models import User
from django.contrib.contenttypes.models import ContentType
from django.contrib.contenttypes.fields import GenericForeignKey
from workspace.models import Workspace
from portfolio.models import Portfolio
from project.models import Project
from board.models import Board
from collaborator.models import Collaborator, Area


class VisibilityPermission(models.Model):
    """
    Modelo para controlar permissões de visão hierárquicas.
    Permite controle granular de acesso a Workspace, Portfolio, Projeto e Board.
    """
    
    PERMISSION_TYPES = [
        ('workspace', 'Workspace'),
        ('portfolio', 'Portfolio'),
        ('project', 'Projeto'),
        ('board', 'Quadro'),
    ]
    
    ACCESS_TYPES = [
        ('allow', 'Permitir'),
        ('deny', 'Negar'),
    ]
    
    SCOPE_TYPES = [
        ('collaborator', 'Colaborador'),
        ('area', 'Área'),
    ]
    
    # Tipo de permissão (workspace, portfolio, project, board)
    permission_type = models.CharField(
        max_length=20,
        choices=PERMISSION_TYPES,
        verbose_name="Tipo de Permissão",
        help_text="Nível hierárquico da permissão"
    )
    
    # Tipo de acesso (allow/deny)
    access_type = models.CharField(
        max_length=10,
        choices=ACCESS_TYPES,
        default='allow',
        verbose_name="Tipo de Acesso",
        help_text="Se permite ou nega o acesso"
    )
    
    # Escopo da permissão (colaborador específico ou área inteira)
    scope_type = models.CharField(
        max_length=20,
        choices=SCOPE_TYPES,
        verbose_name="Tipo de Escopo",
        help_text="Se a permissão é para colaborador específico ou área"
    )
    
    # Colaborador específico (se scope_type = 'collaborator')
    collaborator = models.ForeignKey(
        Collaborator,
        on_delete=models.CASCADE,
        blank=True,
        null=True,
        verbose_name="Colaborador",
        help_text="Colaborador específico para esta permissão"
    )
    
    # Área específica (se scope_type = 'area')
    area = models.ForeignKey(
        Area,
        on_delete=models.CASCADE,
        blank=True,
        null=True,
        verbose_name="Área",
        help_text="Área específica para esta permissão"
    )
    
    # Referências genéricas para os objetos
    content_type = models.ForeignKey(
        ContentType,
        on_delete=models.CASCADE,
        verbose_name="Tipo de Conteúdo"
    )
    
    object_id = models.PositiveIntegerField(
        verbose_name="ID do Objeto"
    )
    
    content_object = GenericForeignKey(
        'content_type',
        'object_id'
    )
    
    # Campos de controle
    inherit_from_parent = models.BooleanField(
        default=True,
        verbose_name="Herdar do Pai",
        help_text="Se deve herdar permissões do nível superior"
    )
    
    override_parent = models.BooleanField(
        default=False,
        verbose_name="Sobrescrever Pai",
        help_text="Se esta permissão sobrescreve a do nível superior"
    )
    
    ativo = models.BooleanField(
        default=True,
        verbose_name="Ativo",
        help_text="Indica se a permissão está ativa"
    )
    
    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name="Criado em"
    )
    
    updated_at = models.DateTimeField(
        auto_now=True,
        verbose_name="Atualizado em"
    )

    class Meta:
        verbose_name = "Permissão de Visão"
        verbose_name_plural = "Permissões de Visão"
        unique_together = [
            'permission_type', 'scope_type', 'collaborator', 'area', 
            'content_type', 'object_id'
        ]
        ordering = ['permission_type', 'scope_type', 'content_type']

    def __str__(self):
        scope_name = self.collaborator.nome if self.collaborator else self.area.nome
        return f"{self.get_permission_type_display()} - {scope_name} - {self.get_access_type_display()}"

    def clean(self):
        """Validação customizada para garantir consistência dos dados."""
        from django.core.exceptions import ValidationError
        
        # Verifica se scope_type e os campos correspondentes estão corretos
        if self.scope_type == 'collaborator' and not self.collaborator:
            raise ValidationError("Colaborador deve ser especificado quando scope_type é 'collaborator'")
        
        if self.scope_type == 'area' and not self.area:
            raise ValidationError("Área deve ser especificada quando scope_type é 'area'")
        
        # Verifica se o content_object é do tipo correto
        if self.content_object:
            expected_type = self.permission_type
            actual_type = self.content_object._meta.model_name
            
            if expected_type != actual_type:
                raise ValidationError(f"Tipo de permissão '{expected_type}' não corresponde ao objeto '{actual_type}'")

    def save(self, *args, **kwargs):
        self.clean()
        super().save(*args, **kwargs)

    @classmethod
    def has_access(cls, user, obj, permission_type=None):
        """
        Verifica se um usuário tem acesso a um objeto específico.
        
        Args:
            user: Usuário Django
            obj: Objeto para verificar acesso (Workspace, Portfolio, Project, Board)
            permission_type: Tipo de permissão (opcional, será inferido do objeto)
        
        Returns:
            bool: True se tem acesso, False caso contrário
        """
        if not user.is_authenticated:
            return False
        
        # Superuser tem acesso total a tudo
        if user.is_superuser:
            return True
        
        # Determina o tipo de permissão baseado no objeto
        if not permission_type:
            permission_type = obj._meta.model_name
        
        # Obtém o colaborador do usuário
        try:
            collaborator = Collaborator.objects.get(user=user, ativo=True)
        except Collaborator.DoesNotExist:
            return False
        
        # Verifica permissões específicas do colaborador
        collaborator_permissions = cls.objects.filter(
            permission_type=permission_type,
            scope_type='collaborator',
            collaborator=collaborator,
            content_type=ContentType.objects.get_for_model(obj),
            object_id=obj.id,
            ativo=True
        )
        
        # Verifica permissões da área do colaborador
        area_permissions = cls.objects.filter(
            permission_type=permission_type,
            scope_type='area',
            area=collaborator.area,
            content_type=ContentType.objects.get_for_model(obj),
            object_id=obj.id,
            ativo=True
        )
        
        # Combina todas as permissões
        all_permissions = list(collaborator_permissions) + list(area_permissions)
        
        if not all_permissions:
            # Se não há permissões específicas, não tem acesso
            return False
        
        # Verifica se há alguma permissão de negação
        deny_permissions = [p for p in all_permissions if p.access_type == 'deny']
        if deny_permissions:
            return False
        
        # Verifica se há permissões de permissão
        allow_permissions = [p for p in all_permissions if p.access_type == 'allow']
        return len(allow_permissions) > 0



    @classmethod
    def get_accessible_objects(cls, user, model_class, permission_type=None):
        """
        Retorna todos os objetos de um tipo específico que o usuário pode acessar.
        
        Args:
            user: Usuário Django
            model_class: Classe do modelo (Workspace, Portfolio, Project, Board)
            permission_type: Tipo de permissão (opcional)
        
        Returns:
            QuerySet: Objetos acessíveis
        """
        if not user.is_authenticated:
            return model_class.objects.none()
        
        # Superuser tem acesso total a tudo
        if user.is_superuser:
            return model_class.objects.all()
        
        if not permission_type:
            permission_type = model_class._meta.model_name
        
        # Obtém o colaborador do usuário
        try:
            collaborator = Collaborator.objects.get(user=user, ativo=True)
        except Collaborator.DoesNotExist:
            return model_class.objects.none()
        
        # Filtra objetos baseado nas permissões específicas
        accessible_ids = set()
        
        # Verifica permissões específicas
        permissions = cls.objects.filter(
            permission_type=permission_type,
            ativo=True
        ).filter(
            models.Q(scope_type='collaborator', collaborator=collaborator) |
            models.Q(scope_type='area', area=collaborator.area)
        )
        
        for permission in permissions:
            if permission.access_type == 'allow':
                accessible_ids.add(permission.object_id)
            elif permission.access_type == 'deny':
                accessible_ids.discard(permission.object_id)
        
        return model_class.objects.filter(id__in=accessible_ids)


class WorkspaceAccess(models.Model):
    """
    Modelo simplificado para controle de acesso a workspaces.
    Permite adicionar colaboradores ou áreas inteiras a um workspace.
    """
    
    workspace = models.ForeignKey(
        Workspace,
        on_delete=models.CASCADE,
        verbose_name="Workspace",
        related_name="access_controls"
    )
    
    collaborator = models.ForeignKey(
        Collaborator,
        on_delete=models.CASCADE,
        blank=True,
        null=True,
        verbose_name="Colaborador",
        help_text="Colaborador específico com acesso ao workspace"
    )
    
    area = models.ForeignKey(
        Area,
        on_delete=models.CASCADE,
        blank=True,
        null=True,
        verbose_name="Área",
        help_text="Área inteira com acesso ao workspace"
    )
    
    can_manage = models.BooleanField(
        default=False,
        verbose_name="Pode Gerenciar",
        help_text="Se pode gerenciar o workspace e seus elementos"
    )
    
    ativo = models.BooleanField(
        default=True,
        verbose_name="Ativo"
    )
    
    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name="Criado em"
    )
    
    updated_at = models.DateTimeField(
        auto_now=True,
        verbose_name="Atualizado em"
    )

    class Meta:
        verbose_name = "Acesso ao Workspace"
        verbose_name_plural = "Acessos ao Workspace"
        unique_together = [
            ['workspace', 'collaborator'],
            ['workspace', 'area']
        ]

    def __str__(self):
        if self.collaborator:
            return f"{self.workspace.nome} - {self.collaborator.nome}"
        return f"{self.workspace.nome} - {self.area.nome}"

    def clean(self):
        from django.core.exceptions import ValidationError
        
        if not self.collaborator and not self.area:
            raise ValidationError("Deve especificar um colaborador ou uma área")
        
        if self.collaborator and self.area:
            raise ValidationError("Deve especificar apenas um colaborador OU uma área")

    def save(self, *args, **kwargs):
        self.clean()
        super().save(*args, **kwargs)
