from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.http import JsonResponse
from django.views.decorators.http import require_http_methods
from django.contrib.contenttypes.models import ContentType
from django.db import transaction

from workspace.models import Workspace
from portfolio.models import Portfolio
from project.models import Project
from board.models import Board
from collaborator.models import Collaborator, Area
from .models import VisibilityPermission, WorkspaceAccess
from .mixins import PermissionContextMixin


@login_required
def permission_dashboard(request):
    """
    Dashboard para gerenciar permissões de visão.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        messages.error(request, "Acesso negado. Apenas administradores podem gerenciar permissões.")
        return redirect('home')
    # Obtém estatísticas de permissões
    total_permissions = VisibilityPermission.objects.count()
    active_permissions = VisibilityPermission.objects.filter(ativo=True).count()
    
    # Permissões por tipo
    workspace_permissions = VisibilityPermission.objects.filter(
        permission_type='workspace'
    ).count()
    portfolio_permissions = VisibilityPermission.objects.filter(
        permission_type='portfolio'
    ).count()
    project_permissions = VisibilityPermission.objects.filter(
        permission_type='project'
    ).count()
    board_permissions = VisibilityPermission.objects.filter(
        permission_type='board'
    ).count()
    
    # Permissões por escopo
    collaborator_permissions = VisibilityPermission.objects.filter(
        scope_type='collaborator'
    ).count()
    area_permissions = VisibilityPermission.objects.filter(
        scope_type='area'
    ).count()
    
    context = {
        'total_permissions': total_permissions,
        'active_permissions': active_permissions,
        'workspace_permissions': workspace_permissions,
        'portfolio_permissions': portfolio_permissions,
        'project_permissions': project_permissions,
        'board_permissions': board_permissions,
        'collaborator_permissions': collaborator_permissions,
        'area_permissions': area_permissions,
    }
    
    return render(request, 'permissions/dashboard.html', context)


@login_required
def workspace_permissions(request, workspace_id):
    """
    Gerencia permissões de um workspace específico.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        messages.error(request, "Acesso negado. Apenas administradores podem gerenciar permissões.")
        return redirect('home')
    
    workspace = get_object_or_404(Workspace, id=workspace_id)
    
    # Obtém permissões existentes
    permissions = VisibilityPermission.objects.filter(
        content_type=ContentType.objects.get_for_model(workspace),
        object_id=workspace.id
    ).select_related('collaborator', 'area')
    
    # Obtém colaboradores e áreas disponíveis
    collaborators = Collaborator.objects.filter(ativo=True)
    areas = Area.objects.filter(ativo=True)
    
    context = {
        'workspace': workspace,
        'permissions': permissions,
        'collaborators': collaborators,
        'areas': areas,
    }
    
    return render(request, 'permissions/workspace_permissions.html', context)


@login_required
@require_http_methods(["POST"])
def add_workspace_permission(request, workspace_id):
    """
    Adiciona uma permissão de acesso ao workspace.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        return JsonResponse({'success': False, 'message': 'Acesso negado. Apenas administradores podem gerenciar permissões.'})
    
    workspace = get_object_or_404(Workspace, id=workspace_id)
    
    scope_type = request.POST.get('scope_type')
    access_type = request.POST.get('access_type', 'allow')
    
    try:
        with transaction.atomic():
            if scope_type == 'collaborator':
                collaborator_id = request.POST.get('collaborator_id')
                collaborator = get_object_or_404(Collaborator, id=collaborator_id)
                
                permission, created = VisibilityPermission.objects.get_or_create(
                    permission_type='workspace',
                    scope_type='collaborator',
                    collaborator=collaborator,
                    content_type=ContentType.objects.get_for_model(workspace),
                    object_id=workspace.id,
                    defaults={
                        'access_type': access_type,
                        'ativo': True
                    }
                )
                
                if not created:
                    permission.access_type = access_type
                    permission.ativo = True
                    permission.save()
                
                message = f"Permissão adicionada para {collaborator.nome}"
                
            elif scope_type == 'area':
                area_id = request.POST.get('area_id')
                area = get_object_or_404(Area, id=area_id)
                
                permission, created = VisibilityPermission.objects.get_or_create(
                    permission_type='workspace',
                    scope_type='area',
                    area=area,
                    content_type=ContentType.objects.get_for_model(workspace),
                    object_id=workspace.id,
                    defaults={
                        'access_type': access_type,
                        'ativo': True
                    }
                )
                
                if not created:
                    permission.access_type = access_type
                    permission.ativo = True
                    permission.save()
                
                message = f"Permissão adicionada para área {area.nome}"
            
            else:
                return JsonResponse({'success': False, 'message': 'Tipo de escopo inválido'})
            
            return JsonResponse({'success': True, 'message': message})
            
    except Exception as e:
        return JsonResponse({'success': False, 'message': str(e)})


@login_required
@require_http_methods(["POST"])
def remove_workspace_permission(request, workspace_id, permission_id):
    """
    Remove uma permissão de acesso ao workspace.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        return JsonResponse({'success': False, 'message': 'Acesso negado. Apenas administradores podem gerenciar permissões.'})
    
    workspace = get_object_or_404(Workspace, id=workspace_id)
    permission = get_object_or_404(VisibilityPermission, id=permission_id)
    
    try:
        permission.ativo = False
        permission.save()
        
        scope_name = permission.collaborator.nome if permission.collaborator else permission.area.nome
        message = f"Permissão removida para {scope_name}"
        
        return JsonResponse({'success': True, 'message': message})
        
    except Exception as e:
        return JsonResponse({'success': False, 'message': str(e)})


@login_required
def bulk_permission_manager(request):
    """
    Gerencia permissões em lote para múltiplos objetos, colaboradores e áreas.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        messages.error(request, "Acesso negado. Apenas administradores podem gerenciar permissões.")
        return redirect('home')
    if request.method == 'POST':
        # Processa adição em lote de permissões
        object_type = request.POST.get('object_type')
        object_ids = request.POST.getlist('object_ids')
        collaborator_ids = request.POST.getlist('collaborator_ids')
        area_ids = request.POST.getlist('area_ids')
        access_type = request.POST.get('access_type', 'allow')
        
        if not object_ids:
            messages.error(request, 'Selecione pelo menos um objeto.')
            return redirect('permissions:bulk_manager')
        
        if not collaborator_ids and not area_ids:
            messages.error(request, 'Selecione pelo menos um colaborador ou área.')
            return redirect('permissions:bulk_manager')
        
        try:
            with transaction.atomic():
                created_count = 0
                
                # Processar colaboradores
                for collaborator_id in collaborator_ids:
                    collaborator = get_object_or_404(Collaborator, id=collaborator_id)
                    
                    for obj_id in object_ids:
                        permission, created = VisibilityPermission.objects.get_or_create(
                            permission_type=object_type,
                            scope_type='collaborator',
                            collaborator=collaborator,
                            content_type=ContentType.objects.get_for_model(
                                get_model_class(object_type)
                            ),
                            object_id=obj_id,
                            defaults={
                                'access_type': access_type,
                                'ativo': True
                            }
                        )
                        
                        if not created:
                            permission.access_type = access_type
                            permission.ativo = True
                            permission.save()
                        
                        if created:
                            created_count += 1
                
                # Processar áreas
                for area_id in area_ids:
                    area = get_object_or_404(Area, id=area_id)
                    
                    for obj_id in object_ids:
                        permission, created = VisibilityPermission.objects.get_or_create(
                            permission_type=object_type,
                            scope_type='area',
                            area=area,
                            content_type=ContentType.objects.get_for_model(
                                get_model_class(object_type)
                            ),
                            object_id=obj_id,
                            defaults={
                                'access_type': access_type,
                                'ativo': True
                            }
                        )
                        
                        if not created:
                            permission.access_type = access_type
                            permission.ativo = True
                            permission.save()
                        
                        if created:
                            created_count += 1
                
                # Mensagem de sucesso
                total_operations = len(collaborator_ids) * len(object_ids) + len(area_ids) * len(object_ids)
                message = f"Operação concluída! {created_count} novas permissões criadas de {total_operations} operações."
                messages.success(request, message)
                return redirect('permissions:bulk_manager')
                
        except Exception as e:
            messages.error(request, f"Erro ao adicionar permissões: {str(e)}")
    
    # Obtém objetos disponíveis para gerenciamento
    workspaces = Workspace.objects.filter(ativo=True)
    portfolios = Portfolio.objects.filter(ativo=True)
    projects = Project.objects.filter(ativo=True)
    boards = Board.objects.filter(ativo=True)
    
    # Obtém colaboradores e áreas
    collaborators = Collaborator.objects.filter(ativo=True)
    areas = Area.objects.filter(ativo=True)
    
    context = {
        'workspaces': workspaces,
        'portfolios': portfolios,
        'projects': projects,
        'boards': boards,
        'collaborators': collaborators,
        'areas': areas,
    }
    
    return render(request, 'permissions/bulk_manager_simple.html', context)





def get_model_class(object_type):
    """
    Retorna a classe do modelo baseado no tipo.
    """
    model_map = {
        'workspace': Workspace,
        'portfolio': Portfolio,
        'project': Project,
        'board': Board,
    }
    return model_map.get(object_type)


# ============================================================================
# VIEWS PARA PORTFOLIO
# ============================================================================

@login_required
def portfolio_permissions(request, portfolio_id):
    """
    Gerencia permissões de um portfolio específico.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        messages.error(request, "Acesso negado. Apenas administradores podem gerenciar permissões.")
        return redirect('home')
    
    portfolio = get_object_or_404(Portfolio, id=portfolio_id)
    
    # Obtém permissões existentes
    permissions = VisibilityPermission.objects.filter(
        content_type=ContentType.objects.get_for_model(portfolio),
        object_id=portfolio.id
    ).select_related('collaborator', 'area')
    
    # Obtém colaboradores e áreas disponíveis
    collaborators = Collaborator.objects.filter(ativo=True)
    areas = Area.objects.filter(ativo=True)
    
    context = {
        'portfolio': portfolio,
        'permissions': permissions,
        'collaborators': collaborators,
        'areas': areas,
    }
    
    return render(request, 'permissions/portfolio_permissions.html', context)


@login_required
@require_http_methods(["POST"])
def add_portfolio_permission(request, portfolio_id):
    """
    Adiciona uma permissão de acesso ao portfolio.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        return JsonResponse({'success': False, 'message': 'Acesso negado. Apenas administradores podem gerenciar permissões.'})
    
    portfolio = get_object_or_404(Portfolio, id=portfolio_id)
    
    scope_type = request.POST.get('scope_type')
    access_type = request.POST.get('access_type', 'allow')
    
    try:
        with transaction.atomic():
            if scope_type == 'collaborator':
                collaborator_id = request.POST.get('collaborator_id')
                collaborator = get_object_or_404(Collaborator, id=collaborator_id)
                
                permission, created = VisibilityPermission.objects.get_or_create(
                    permission_type='portfolio',
                    scope_type='collaborator',
                    collaborator=collaborator,
                    content_type=ContentType.objects.get_for_model(portfolio),
                    object_id=portfolio.id,
                    defaults={
                        'access_type': access_type,
                        'ativo': True
                    }
                )
                
                if not created:
                    permission.access_type = access_type
                    permission.ativo = True
                    permission.save()
                
                message = f"Permissão adicionada para {collaborator.nome}"
                
            elif scope_type == 'area':
                area_id = request.POST.get('area_id')
                area = get_object_or_404(Area, id=area_id)
                
                permission, created = VisibilityPermission.objects.get_or_create(
                    permission_type='portfolio',
                    scope_type='area',
                    area=area,
                    content_type=ContentType.objects.get_for_model(portfolio),
                    object_id=portfolio.id,
                    defaults={
                        'access_type': access_type,
                        'ativo': True
                    }
                )
                
                if not created:
                    permission.access_type = access_type
                    permission.ativo = True
                    permission.save()
                
                message = f"Permissão adicionada para área {area.nome}"
            
            else:
                return JsonResponse({'success': False, 'message': 'Tipo de escopo inválido'})
            
            return JsonResponse({'success': True, 'message': message})
            
    except Exception as e:
        return JsonResponse({'success': False, 'message': str(e)})


@login_required
@require_http_methods(["POST"])
def remove_portfolio_permission(request, portfolio_id, permission_id):
    """
    Remove uma permissão de acesso ao portfolio.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        return JsonResponse({'success': False, 'message': 'Acesso negado. Apenas administradores podem gerenciar permissões.'})
    
    portfolio = get_object_or_404(Portfolio, id=portfolio_id)
    permission = get_object_or_404(VisibilityPermission, id=permission_id)
    
    try:
        permission.ativo = False
        permission.save()
        
        scope_name = permission.collaborator.nome if permission.collaborator else permission.area.nome
        message = f"Permissão removida para {scope_name}"
        
        return JsonResponse({'success': True, 'message': message})
        
    except Exception as e:
        return JsonResponse({'success': False, 'message': str(e)})


# ============================================================================
# VIEWS PARA PROJECT
# ============================================================================

@login_required
def project_permissions(request, project_id):
    """
    Gerencia permissões de um project específico.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        messages.error(request, "Acesso negado. Apenas administradores podem gerenciar permissões.")
        return redirect('home')
    
    project = get_object_or_404(Project, id=project_id)
    
    # Obtém permissões existentes
    permissions = VisibilityPermission.objects.filter(
        content_type=ContentType.objects.get_for_model(project),
        object_id=project.id
    ).select_related('collaborator', 'area')
    
    # Obtém colaboradores e áreas disponíveis
    collaborators = Collaborator.objects.filter(ativo=True)
    areas = Area.objects.filter(ativo=True)
    
    context = {
        'project': project,
        'permissions': permissions,
        'collaborators': collaborators,
        'areas': areas,
    }
    
    return render(request, 'permissions/project_permissions.html', context)


@login_required
@require_http_methods(["POST"])
def add_project_permission(request, project_id):
    """
    Adiciona uma permissão de acesso ao project.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        return JsonResponse({'success': False, 'message': 'Acesso negado. Apenas administradores podem gerenciar permissões.'})
    
    project = get_object_or_404(Project, id=project_id)
    
    scope_type = request.POST.get('scope_type')
    access_type = request.POST.get('access_type', 'allow')
    
    try:
        with transaction.atomic():
            if scope_type == 'collaborator':
                collaborator_id = request.POST.get('collaborator_id')
                collaborator = get_object_or_404(Collaborator, id=collaborator_id)
                
                permission, created = VisibilityPermission.objects.get_or_create(
                    permission_type='project',
                    scope_type='collaborator',
                    collaborator=collaborator,
                    content_type=ContentType.objects.get_for_model(project),
                    object_id=project.id,
                    defaults={
                        'access_type': access_type,
                        'ativo': True
                    }
                )
                
                if not created:
                    permission.access_type = access_type
                    permission.ativo = True
                    permission.save()
                
                message = f"Permissão adicionada para {collaborator.nome}"
                
            elif scope_type == 'area':
                area_id = request.POST.get('area_id')
                area = get_object_or_404(Area, id=area_id)
                
                permission, created = VisibilityPermission.objects.get_or_create(
                    permission_type='project',
                    scope_type='area',
                    area=area,
                    content_type=ContentType.objects.get_for_model(project),
                    object_id=project.id,
                    defaults={
                        'access_type': access_type,
                        'ativo': True
                    }
                )
                
                if not created:
                    permission.access_type = access_type
                    permission.ativo = True
                    permission.save()
                
                message = f"Permissão adicionada para área {area.nome}"
            
            else:
                return JsonResponse({'success': False, 'message': 'Tipo de escopo inválido'})
            
            return JsonResponse({'success': True, 'message': message})
            
    except Exception as e:
        return JsonResponse({'success': False, 'message': str(e)})


@login_required
@require_http_methods(["POST"])
def remove_project_permission(request, project_id, permission_id):
    """
    Remove uma permissão de acesso ao project.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        return JsonResponse({'success': False, 'message': 'Acesso negado. Apenas administradores podem gerenciar permissões.'})
    
    project = get_object_or_404(Project, id=project_id)
    permission = get_object_or_404(VisibilityPermission, id=permission_id)
    
    try:
        permission.ativo = False
        permission.save()
        
        scope_name = permission.collaborator.nome if permission.collaborator else permission.area.nome
        message = f"Permissão removida para {scope_name}"
        
        return JsonResponse({'success': True, 'message': message})
        
    except Exception as e:
        return JsonResponse({'success': False, 'message': str(e)})


# ============================================================================
# VIEWS PARA BOARD
# ============================================================================

@login_required
def board_permissions(request, board_id):
    """
    Gerencia permissões de um board específico.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        messages.error(request, "Acesso negado. Apenas administradores podem gerenciar permissões.")
        return redirect('home')
    
    board = get_object_or_404(Board, id=board_id)
    
    # Obtém permissões existentes
    permissions = VisibilityPermission.objects.filter(
        content_type=ContentType.objects.get_for_model(board),
        object_id=board.id
    ).select_related('collaborator', 'area')
    
    # Obtém colaboradores e áreas disponíveis
    collaborators = Collaborator.objects.filter(ativo=True)
    areas = Area.objects.filter(ativo=True)
    
    context = {
        'board': board,
        'permissions': permissions,
        'collaborators': collaborators,
        'areas': areas,
    }
    
    return render(request, 'permissions/board_permissions.html', context)


@login_required
@require_http_methods(["POST"])
def add_board_permission(request, board_id):
    """
    Adiciona uma permissão de acesso ao board.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        return JsonResponse({'success': False, 'message': 'Acesso negado. Apenas administradores podem gerenciar permissões.'})
    
    board = get_object_or_404(Board, id=board_id)
    
    scope_type = request.POST.get('scope_type')
    access_type = request.POST.get('access_type', 'allow')
    
    try:
        with transaction.atomic():
            if scope_type == 'collaborator':
                collaborator_id = request.POST.get('collaborator_id')
                collaborator = get_object_or_404(Collaborator, id=collaborator_id)
                
                permission, created = VisibilityPermission.objects.get_or_create(
                    permission_type='board',
                    scope_type='collaborator',
                    collaborator=collaborator,
                    content_type=ContentType.objects.get_for_model(board),
                    object_id=board.id,
                    defaults={
                        'access_type': access_type,
                        'ativo': True
                    }
                )
                
                if not created:
                    permission.access_type = access_type
                    permission.ativo = True
                    permission.save()
                
                message = f"Permissão adicionada para {collaborator.nome}"
                
            elif scope_type == 'area':
                area_id = request.POST.get('area_id')
                area = get_object_or_404(Area, id=area_id)
                
                permission, created = VisibilityPermission.objects.get_or_create(
                    permission_type='board',
                    scope_type='area',
                    area=area,
                    content_type=ContentType.objects.get_for_model(board),
                    object_id=board.id,
                    defaults={
                        'access_type': access_type,
                        'ativo': True
                    }
                )
                
                if not created:
                    permission.access_type = access_type
                    permission.ativo = True
                    permission.save()
                
                message = f"Permissão adicionada para área {area.nome}"
            
            else:
                return JsonResponse({'success': False, 'message': 'Tipo de escopo inválido'})
            
            return JsonResponse({'success': True, 'message': message})
            
    except Exception as e:
        return JsonResponse({'success': False, 'message': str(e)})


@login_required
@require_http_methods(["POST"])
def remove_board_permission(request, board_id, permission_id):
    """
    Remove uma permissão de acesso ao board.
    """
    # Verifica se o usuário é administrador
    if not request.user.is_superuser:
        return JsonResponse({'success': False, 'message': 'Acesso negado. Apenas administradores podem gerenciar permissões.'})
    board = get_object_or_404(Board, id=board_id)
    permission = get_object_or_404(VisibilityPermission, id=permission_id)
    
    try:
        permission.ativo = False
        permission.save()
        
        scope_name = permission.collaborator.nome if permission.collaborator else permission.area.nome
        message = f"Permissão removida para {scope_name}"
        
        return JsonResponse({'success': True, 'message': message})
        
    except Exception as e:
        return JsonResponse({'success': False, 'message': str(e)})
