from rest_framework import viewsets, status
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter

from .models import Portfolio
from .serializers import (
    PortfolioSerializer, 
    PortfolioDetailSerializer, 
    PortfolioListSerializer,
    PortfolioCreateSerializer
)


class PortfolioViewSet(viewsets.ModelViewSet):
    """
    ViewSet para operações CRUD de portfolios.
    """
    
    queryset = Portfolio.objects.all()
    permission_classes = [IsAuthenticated]
    filter_backends = [DjangoFilterBackend, SearchFilter, OrderingFilter]
    filterset_fields = ['ativo', 'workspace', 'cliente', 'status', 'risco', 'prioridade']
    search_fields = ['nome', 'descricao', 'workspace__nome', 'cliente__nome']
    ordering_fields = ['nome', 'workspace__nome', 'prazo', 'progresso', 'created_at']
    ordering = ['workspace__nome', 'nome']

    def get_serializer_class(self):
        if self.action == 'create':
            return PortfolioCreateSerializer
        if self.action == 'retrieve':
            return PortfolioDetailSerializer
        if self.action == 'list':
            return PortfolioListSerializer
        return PortfolioSerializer

    def get_queryset(self):
        queryset = Portfolio.objects.select_related('workspace', 'cliente').all()
        
        # Filtro por status ativo
        ativo = self.request.query_params.get('ativo', None)
        if ativo is not None:
            queryset = queryset.filter(ativo=ativo.lower() == 'true')
            
        # Filtro por workspace
        workspace_id = self.request.query_params.get('workspace_id', None)
        if workspace_id:
            queryset = queryset.filter(workspace_id=workspace_id)
            
        # Filtro por cliente
        cliente_id = self.request.query_params.get('cliente_id', None)
        if cliente_id:
            queryset = queryset.filter(cliente_id=cliente_id)
            
        return queryset

    @action(detail=True, methods=['post'])
    def toggle_status(self, request, pk=None):
        """
        Alterna o status ativo/inativo do portfolio.
        """
        portfolio = self.get_object()
        portfolio.ativo = not portfolio.ativo
        portfolio.save()
        
        serializer = self.get_serializer(portfolio)
        return Response(serializer.data)

    @action(detail=True, methods=['get'])
    def projects(self, request, pk=None):
        """
        Retorna os projetos do portfolio.
        """
        portfolio = self.get_object()
        projects = portfolio.projects.filter(ativo=True)
        
        from project.serializers import ProjectListSerializer
        serializer = ProjectListSerializer(projects, many=True)
        return Response(serializer.data)

    @action(detail=True, methods=['get'])
    def burndown(self, request, pk=None):
        """
        Retorna dados do gráfico de burndown do portfolio.
        """
        portfolio = self.get_object()
        burndown_data = portfolio.burndown_chart_data
        return Response(burndown_data)

    @action(detail=True, methods=['patch'])
    def update_progress(self, request, pk=None):
        """
        Atualiza o progresso do portfolio.
        """
        portfolio = self.get_object()
        progresso = request.data.get('progresso')
        
        if progresso is not None:
            try:
                progresso = int(progresso)
                if 0 <= progresso <= 100:
                    portfolio.progresso = progresso
                    portfolio.save()
                    
                    serializer = self.get_serializer(portfolio)
                    return Response(serializer.data)
                else:
                    return Response(
                        {"error": "Progresso deve estar entre 0 e 100"}, 
                        status=status.HTTP_400_BAD_REQUEST
                    )
            except ValueError:
                return Response(
                    {"error": "Progresso deve ser um número inteiro"}, 
                    status=status.HTTP_400_BAD_REQUEST
                )
        
        return Response(
            {"error": "Campo 'progresso' é obrigatório"}, 
            status=status.HTTP_400_BAD_REQUEST
        )

    @action(detail=True, methods=['get'])
    def statistics(self, request, pk=None):
        """
        Retorna estatísticas do portfolio.
        """
        portfolio = self.get_object()
        
        stats = {
            'total_projects': portfolio.total_projects,
            'projects_concluidos': portfolio.projects_concluidos,
            'burndown_data': portfolio.burndown_chart_data,
            'projects_por_status': {},
            'projects_por_prioridade': {},
        }
        
        # Estatísticas de projetos por status
        for project in portfolio.projects.all():
            status_key = project.status
            if status_key not in stats['projects_por_status']:
                stats['projects_por_status'][status_key] = 0
            stats['projects_por_status'][status_key] += 1
        
        # Estatísticas de projetos por prioridade
        for project in portfolio.projects.all():
            prioridade_key = project.prioridade
            if prioridade_key not in stats['projects_por_prioridade']:
                stats['projects_por_prioridade'][prioridade_key] = 0
            stats['projects_por_prioridade'][prioridade_key] += 1
        
        return Response(stats)

