from django.db import models
from django.core.validators import MinValueValidator, MaxValueValidator
from client.models import Client
from workspace.models import Workspace


class Portfolio(models.Model):
    """
    Modelo para representar portfolios de projetos.
    Um portfolio agrupa projetos relacionados dentro de um workspace.
    """
    
    STATUS_CHOICES = [
        ('planejamento', 'Planejamento'),
        ('em_andamento', 'Em Andamento'),
        ('pausado', 'Pausado'),
        ('concluido', 'Concluído'),
        ('cancelado', 'Cancelado'),
    ]
    
    RISCO_CHOICES = [
        ('baixo', 'Baixo'),
        ('medio', 'Médio'),
        ('alto', 'Alto'),
        ('critico', 'Crítico'),
    ]
    
    PRIORIDADE_CHOICES = [
        ('baixa', 'Baixa'),
        ('media', 'Média'),
        ('alta', 'Alta'),
        ('urgente', 'Urgente'),
    ]
    
    nome = models.CharField(
        max_length=200,
        verbose_name="Nome",
        help_text="Nome do portfolio"
    )
    
    workspace = models.ForeignKey(
        Workspace,
        on_delete=models.CASCADE,
        verbose_name="Workspace",
        help_text="Workspace ao qual o portfolio pertence",
        related_name="portfolios"
    )
    
    cliente = models.ForeignKey(
        Client,
        on_delete=models.CASCADE,
        verbose_name="Cliente",
        help_text="Cliente proprietário do portfolio",
        related_name="portfolios"
    )
    
    descricao = models.TextField(
        blank=True,
        null=True,
        verbose_name="Descrição",
        help_text="Descrição do portfolio"
    )
    
    # Campos gerenciais
    status = models.CharField(
        max_length=20,
        choices=STATUS_CHOICES,
        default='planejamento',
        verbose_name="Status",
        help_text="Status atual do portfolio"
    )
    
    risco = models.CharField(
        max_length=10,
        choices=RISCO_CHOICES,
        default='baixo',
        verbose_name="Risco",
        help_text="Nível de risco do portfolio"
    )
    
    prazo = models.DateField(
        blank=True,
        null=True,
        verbose_name="Prazo",
        help_text="Data limite para conclusão do portfolio"
    )
    
    progresso = models.IntegerField(
        default=0,
        validators=[MinValueValidator(0), MaxValueValidator(100)],
        verbose_name="Progresso (%)",
        help_text="Percentual de progresso do portfolio (0-100)"
    )
    
    prioridade = models.CharField(
        max_length=10,
        choices=PRIORIDADE_CHOICES,
        default='media',
        verbose_name="Prioridade",
        help_text="Prioridade do portfolio"
    )
    
    rentabilidade = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        blank=True,
        null=True,
        verbose_name="Rentabilidade",
        help_text="Rentabilidade esperada/realizada do portfolio"
    )
    
    orcamento = models.DecimalField(
        max_digits=12,
        decimal_places=2,
        blank=True,
        null=True,
        verbose_name="Orçamento",
        help_text="Orçamento total do portfolio"
    )
    
    ativo = models.BooleanField(
        default=True,
        verbose_name="Ativo",
        help_text="Indica se o portfolio está ativo"
    )
    
    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name="Criado em"
    )
    
    updated_at = models.DateTimeField(
        auto_now=True,
        verbose_name="Atualizado em"
    )

    class Meta:
        verbose_name = "Portfolio"
        verbose_name_plural = "Portfolios"
        ordering = ['workspace', 'nome']
        unique_together = ['workspace', 'nome']

    def __str__(self):
        return f"{self.workspace.nome} - {self.nome}"

    def get_absolute_url(self):
        from django.urls import reverse
        return reverse('portfolio:detail', kwargs={'pk': self.pk})

    @property
    def total_projects(self):
        return self.projects.count()

    @property
    def projects_concluidos(self):
        return self.projects.filter(status='concluido').count()

    @property
    def burndown_chart_data(self):
        """
        Retorna dados para gráfico de burndown do portfolio.
        """
        # Implementação simplificada - pode ser expandida
        projects = self.projects.all()
        total_tasks = sum(project.total_tasks for project in projects)
        completed_tasks = sum(project.tasks_concluidas for project in projects)
        
        return {
            'total_tasks': total_tasks,
            'completed_tasks': completed_tasks,
            'remaining_tasks': total_tasks - completed_tasks,
            'completion_rate': (completed_tasks / total_tasks * 100) if total_tasks > 0 else 0
        }

    def save(self, *args, **kwargs):
        # Garante que o cliente do portfolio seja o mesmo do workspace
        if self.workspace:
            self.cliente = self.workspace.cliente
        super().save(*args, **kwargs)

