from rest_framework import serializers
from .models import Portfolio
from workspace.serializers import WorkspaceListSerializer
from client.serializers import ClientListSerializer


class PortfolioSerializer(serializers.ModelSerializer):
    """
    Serializer para o modelo Portfolio.
    """
    
    workspace_nome = serializers.CharField(source='workspace.nome', read_only=True)
    cliente_nome = serializers.CharField(source='cliente.nome', read_only=True)
    total_projects = serializers.ReadOnlyField()
    projects_concluidos = serializers.ReadOnlyField()
    burndown_chart_data = serializers.ReadOnlyField()
    
    class Meta:
        model = Portfolio
        fields = [
            'id', 'nome', 'workspace', 'workspace_nome', 'cliente', 'cliente_nome',
            'descricao', 'status', 'risco', 'prazo', 'progresso', 'prioridade',
            'rentabilidade', 'orcamento', 'ativo',
            'total_projects', 'projects_concluidos', 'burndown_chart_data',
            'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'cliente', 'created_at', 'updated_at']


class PortfolioDetailSerializer(serializers.ModelSerializer):
    """
    Serializer detalhado para o modelo Portfolio.
    """
    
    workspace = WorkspaceListSerializer(read_only=True)
    cliente = ClientListSerializer(read_only=True)
    projects = serializers.SerializerMethodField()
    burndown_chart_data = serializers.ReadOnlyField()
    
    class Meta:
        model = Portfolio
        fields = [
            'id', 'nome', 'workspace', 'cliente', 'descricao',
            'status', 'risco', 'prazo', 'progresso', 'prioridade',
            'rentabilidade', 'orcamento', 'ativo',
            'projects', 'burndown_chart_data',
            'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'cliente', 'created_at', 'updated_at']
    
    def get_projects(self, obj):
        from project.serializers import ProjectListSerializer
        projects = obj.projects.filter(ativo=True)
        return ProjectListSerializer(projects, many=True).data


class PortfolioListSerializer(serializers.ModelSerializer):
    """
    Serializer simplificado para listagem de portfolios.
    """
    
    workspace_nome = serializers.CharField(source='workspace.nome', read_only=True)
    cliente_nome = serializers.CharField(source='cliente.nome', read_only=True)
    total_projects = serializers.ReadOnlyField()
    
    class Meta:
        model = Portfolio
        fields = [
            'id', 'nome', 'workspace_nome', 'cliente_nome',
            'status', 'risco', 'prazo', 'progresso', 'prioridade',
            'total_projects', 'ativo'
        ]


class PortfolioCreateSerializer(serializers.ModelSerializer):
    """
    Serializer para criação de portfolios.
    """
    
    class Meta:
        model = Portfolio
        fields = [
            'nome', 'workspace', 'descricao', 'status', 'risco',
            'prazo', 'progresso', 'prioridade', 'rentabilidade', 'orcamento'
        ]
    
    def create(self, validated_data):
        # O cliente é automaticamente definido pelo workspace
        workspace = validated_data['workspace']
        validated_data['cliente'] = workspace.cliente
        return super().create(validated_data)

