from django.shortcuts import render, get_object_or_404, redirect
from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.core.paginator import Paginator
from django.db.models import Q, Count
from django.http import JsonResponse
from django.views.generic import CreateView, UpdateView
from django.contrib.auth.mixins import LoginRequiredMixin
from django.urls import reverse_lazy
from .models import Portfolio
from .forms import PortfolioForm
from workspace.models import Workspace
from client.models import Client
from gestao_agencia.decorators import superuser_required
from gestao_agencia.mixins import SuperuserRequiredMixin


@login_required
def portfolio_list(request):
    """
    Lista todos os portfolios com filtros e paginação.
    """
    # Obter parâmetros de filtro
    search = request.GET.get('search', '')
    status = request.GET.get('status', '')
    workspace_id = request.GET.get('workspace', '')
    
    # Query base
    portfolios = Portfolio.objects.select_related('workspace', 'cliente').all()
    
    # Aplicar filtros
    if search:
        portfolios = portfolios.filter(
            Q(nome__icontains=search) |
            Q(workspace__nome__icontains=search) |
            Q(cliente__nome__icontains=search) |
            Q(descricao__icontains=search)
        )
    
    if status:
        portfolios = portfolios.filter(status=status)
    
    if workspace_id:
        portfolios = portfolios.filter(workspace_id=workspace_id)
    
    # Ordenar por workspace e nome
    portfolios = portfolios.order_by('workspace__nome', 'nome')
    
    # Paginação
    paginator = Paginator(portfolios, 12)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)
    
    # Estatísticas
    total_portfolios = Portfolio.objects.count()
    portfolios_ativos = Portfolio.objects.filter(ativo=True).count()
    portfolios_em_andamento = Portfolio.objects.filter(status='em_andamento').count()
    portfolios_concluidos = Portfolio.objects.filter(status='concluido').count()
    
    # Lista de workspaces para o filtro
    workspaces = Workspace.objects.all().order_by('nome')
    
    context = {
        'portfolios': page_obj,
        'page_obj': page_obj,
        'is_paginated': page_obj.has_other_pages(),
        'search': search,
        'status': status,
        'workspace_id': workspace_id,
        'total_portfolios': total_portfolios,
        'portfolios_ativos': portfolios_ativos,
        'portfolios_em_andamento': portfolios_em_andamento,
        'portfolios_concluidos': portfolios_concluidos,
        'workspaces': workspaces,
    }
    
    return render(request, 'portfolio/list.html', context)





@login_required
def portfolio_detail(request, pk):
    """
    Exibe os detalhes de um portfolio.
    """
    portfolio = get_object_or_404(Portfolio, pk=pk)
    
    context = {
        'portfolio': portfolio,
    }
    
    return render(request, 'portfolio/detail.html', context)





@login_required
@superuser_required
def portfolio_delete(request, pk):
    """
    Exclui um portfolio.
    """
    portfolio = get_object_or_404(Portfolio, pk=pk)
    
    if request.method == 'POST':
        nome = portfolio.nome
        portfolio.delete()
        messages.success(request, f'Portfolio "{nome}" excluído com sucesso!')
        return redirect('portfolio:list')
    
    context = {
        'portfolio': portfolio,
    }
    
    return render(request, 'portfolio/delete.html', context)


class PortfolioCreateModalView(LoginRequiredMixin, SuperuserRequiredMixin, CreateView):
    """
    View para criação de portfolio via modal.
    """
    model = Portfolio
    form_class = PortfolioForm
    template_name = 'portfolio/modal_create.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context.update({
            'modal_id': 'createPortfolioModal',
            'modal_title': 'Novo Portfolio',
            'modal_icon': 'bi-collection',
            'form_id': 'portfolioForm',
            'submit_text': 'Criar Portfolio'
        })
        return context
    
    def form_valid(self, form):
        try:
            portfolio = form.save(commit=False)
            # Garante que o cliente seja definido antes de salvar
            if portfolio.workspace:
                portfolio.cliente = portfolio.workspace.cliente
            portfolio.save()
            
            if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
                return JsonResponse({
                    'success': True,
                    'message': 'Portfolio criado com sucesso!',
                    'obj': {
                        'id': portfolio.id,
                        'nome': portfolio.nome,
                        'detail_url': reverse_lazy('portfolio:detail', kwargs={'pk': portfolio.id})
                    }
                })
            return super().form_valid(form)
        except Exception as e:
            print(f"Erro ao criar portfolio: {e}")
            if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
                return JsonResponse({
                    'success': False,
                    'errors': {'__all__': [str(e)]}
                })
            raise
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)


class PortfolioUpdateModalView(LoginRequiredMixin, SuperuserRequiredMixin, UpdateView):
    """
    View para edição de portfolio via modal.
    """
    model = Portfolio
    form_class = PortfolioForm
    template_name = 'portfolio/modal_edit.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        portfolio = self.get_object()
        context.update({
            'modal_id': 'editPortfolioModal',
            'modal_title': f'Editar Portfolio: {portfolio.nome}',
            'modal_icon': 'bi-collection-gear',
            'form_id': 'portfolioEditForm',
            'submit_text': 'Atualizar Portfolio'
        })
        return context
    
    def form_valid(self, form):
        try:
            portfolio = form.save(commit=False)
            # Garante que o cliente seja definido antes de salvar
            if portfolio.workspace:
                portfolio.cliente = portfolio.workspace.cliente
            portfolio.save()
            
            if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
                return JsonResponse({
                    'success': True,
                    'message': 'Portfolio atualizado com sucesso!',
                    'obj': {
                        'id': portfolio.id,
                        'nome': portfolio.nome,
                        'status': portfolio.status,
                        'ativo': portfolio.ativo,
                        'detail_url': reverse_lazy('portfolio:detail', kwargs={'pk': portfolio.id})
                    }
                })
            return super().form_valid(form)
        except Exception as e:
            print(f"Erro ao atualizar portfolio: {e}")
            if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
                return JsonResponse({
                    'success': False,
                    'errors': {'__all__': [str(e)]}
                })
            raise
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)
