from rest_framework import viewsets, status
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter

from .models import Project
from .serializers import (
    ProjectSerializer, 
    ProjectDetailSerializer, 
    ProjectListSerializer,
    ProjectCreateSerializer,
    ProjectUpdateStatusSerializer
)


class ProjectViewSet(viewsets.ModelViewSet):
    """
    ViewSet para operações CRUD de projetos.
    """
    
    queryset = Project.objects.all()
    permission_classes = [IsAuthenticated]
    filter_backends = [DjangoFilterBackend, SearchFilter, OrderingFilter]
    filterset_fields = ['ativo', 'portfolio', 'workspace', 'cliente', 'status', 'prioridade']
    search_fields = ['nome', 'descricao', 'portfolio__nome', 'workspace__nome', 'cliente__nome']
    ordering_fields = ['nome', 'portfolio__nome', 'cronograma_inicio', 'cronograma_fim', 'progresso', 'created_at']
    ordering = ['portfolio__nome', 'cronograma_inicio']

    def get_serializer_class(self):
        if self.action == 'create':
            return ProjectCreateSerializer
        if self.action == 'retrieve':
            return ProjectDetailSerializer
        if self.action == 'list':
            return ProjectListSerializer
        if self.action == 'update_status':
            return ProjectUpdateStatusSerializer
        return ProjectSerializer

    def get_queryset(self):
        queryset = Project.objects.select_related('portfolio', 'workspace', 'cliente').all()
        
        # Filtro por status ativo
        ativo = self.request.query_params.get('ativo', None)
        if ativo is not None:
            queryset = queryset.filter(ativo=ativo.lower() == 'true')
            
        # Filtro por portfolio
        portfolio_id = self.request.query_params.get('portfolio_id', None)
        if portfolio_id:
            queryset = queryset.filter(portfolio_id=portfolio_id)
            
        # Filtro por workspace
        workspace_id = self.request.query_params.get('workspace_id', None)
        if workspace_id:
            queryset = queryset.filter(workspace_id=workspace_id)
            
        # Filtro por cliente
        cliente_id = self.request.query_params.get('cliente_id', None)
        if cliente_id:
            queryset = queryset.filter(cliente_id=cliente_id)
            
        # Filtro por projetos atrasados
        atrasados = self.request.query_params.get('atrasados', None)
        if atrasados and atrasados.lower() == 'true':
            from django.utils import timezone
            queryset = queryset.filter(
                cronograma_fim__lt=timezone.now().date(),
                status__in=['planejamento', 'em_andamento', 'pausado']
            )
            
        return queryset

    @action(detail=True, methods=['post'])
    def toggle_status(self, request, pk=None):
        """
        Alterna o status ativo/inativo do projeto.
        """
        project = self.get_object()
        project.ativo = not project.ativo
        project.save()
        
        serializer = self.get_serializer(project)
        return Response(serializer.data)

    @action(detail=True, methods=['patch'])
    def update_status(self, request, pk=None):
        """
        Atualiza o status e informações relacionadas do projeto.
        """
        project = self.get_object()
        serializer = ProjectUpdateStatusSerializer(project, data=request.data, partial=True)
        
        if serializer.is_valid():
            serializer.save()
            return Response(ProjectSerializer(project).data)
        
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    @action(detail=True, methods=['get'])
    def boards(self, request, pk=None):
        """
        Retorna os boards do projeto.
        """
        project = self.get_object()
        boards = project.boards.filter(ativo=True).order_by('ordem', 'nome')
        
        from board.serializers import BoardListSerializer
        serializer = BoardListSerializer(boards, many=True)
        return Response(serializer.data)

    @action(detail=True, methods=['get'])
    def tasks(self, request, pk=None):
        """
        Retorna todas as tarefas do projeto.
        """
        project = self.get_object()
        
        from task.models import Task
        tasks = Task.objects.filter(group__board__project=project)
        
        from task.serializers import TaskListSerializer
        serializer = TaskListSerializer(tasks, many=True)
        return Response(serializer.data)

    @action(detail=True, methods=['get'])
    def timeline(self, request, pk=None):
        """
        Retorna dados para timeline/cronograma do projeto.
        """
        project = self.get_object()
        
        from task.models import Task
        tasks = Task.objects.filter(group__board__project=project).order_by('cronograma_inicio')
        
        timeline_data = []
        for task in tasks:
            timeline_data.append({
                'id': task.id,
                'nome': task.nome,
                'inicio': task.cronograma_inicio,
                'fim': task.cronograma_fim,
                'status': task.status,
                'responsavel': task.responsavel.nome if task.responsavel else None,
                'progresso': task.progresso_percentual,
                'group': task.group.nome,
                'board': task.group.board.nome,
            })
        
        return Response(timeline_data)

    @action(detail=True, methods=['get'])
    def statistics(self, request, pk=None):
        """
        Retorna estatísticas do projeto.
        """
        project = self.get_object()
        
        stats = {
            'duracao_planejada': project.duracao_planejada,
            'duracao_real': project.duracao_real,
            'total_boards': project.total_boards,
            'total_tasks': project.total_tasks,
            'tasks_concluidas': project.tasks_concluidas,
            'is_atrasado': project.is_atrasado,
            'orcamento': float(project.orcamento) if project.orcamento else 0,
            'custo_real': float(project.custo_real),
            'tasks_por_status': {},
            'tasks_por_prioridade': {},
        }
        
        # Estatísticas de tarefas por status
        from task.models import Task
        tasks = Task.objects.filter(group__board__project=project)
        
        for task in tasks:
            status_key = task.status
            if status_key not in stats['tasks_por_status']:
                stats['tasks_por_status'][status_key] = 0
            stats['tasks_por_status'][status_key] += 1
            
            prioridade_key = task.prioridade
            if prioridade_key not in stats['tasks_por_prioridade']:
                stats['tasks_por_prioridade'][prioridade_key] = 0
            stats['tasks_por_prioridade'][prioridade_key] += 1
        
        return Response(stats)

