from django import forms
from .models import Project
from portfolio.models import Portfolio


class ProjectCreateForm(forms.ModelForm):
    """
    Formulário para criação de projetos (sem campos automáticos).
    """
    
    class Meta:
        model = Project
        fields = [
            'nome', 'descricao', 'portfolio', 'status', 'prioridade',
            'cronograma_inicio', 'cronograma_fim', 'orcamento', 'ativo'
        ]
        widgets = {
            'nome': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': 'Nome do projeto'
            }),
            'descricao': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 3,
                'placeholder': 'Descrição do projeto'
            }),
            'portfolio': forms.Select(attrs={
                'class': 'form-select'
            }),
            'status': forms.Select(attrs={
                'class': 'form-select'
            }),
            'prioridade': forms.Select(attrs={
                'class': 'form-select'
            }),
            'cronograma_inicio': forms.DateInput(attrs={
                'class': 'form-control',
                'type': 'date'
            }),
            'cronograma_fim': forms.DateInput(attrs={
                'class': 'form-control',
                'type': 'date'
            }),
            'orcamento': forms.NumberInput(attrs={
                'class': 'form-control',
                'step': '0.01',
                'min': '0'
            }),
            'ativo': forms.CheckboxInput(attrs={
                'class': 'form-check-input'
            })
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        
        # Filtrar apenas portfolios ativos
        self.fields['portfolio'].queryset = Portfolio.objects.filter(ativo=True).order_by('nome')
        
        # Adiciona classes CSS aos labels
        for field_name, field in self.fields.items():
            if field_name != 'ativo':  # Não adicionar form-control ao checkbox
                field.widget.attrs.update({'class': field.widget.attrs.get('class', '') + ' form-control'})


class ProjectForm(forms.ModelForm):
    """
    Formulário completo para edição de projetos (com todos os campos).
    """
    
    class Meta:
        model = Project
        fields = [
            'nome', 'descricao', 'portfolio', 'cliente', 'status', 'prioridade',
            'cronograma_inicio', 'cronograma_fim', 'data_inicio_real', 'data_fim_real',
            'progresso', 'orcamento', 'custo_real', 'ativo'
        ]
        widgets = {
            'nome': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': 'Nome do projeto'
            }),
            'descricao': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 3,
                'placeholder': 'Descrição do projeto'
            }),
            'portfolio': forms.Select(attrs={
                'class': 'form-select'
            }),
            'cliente': forms.Select(attrs={
                'class': 'form-select',
                'readonly': 'readonly'
            }),
            'status': forms.Select(attrs={
                'class': 'form-select'
            }),
            'prioridade': forms.Select(attrs={
                'class': 'form-select'
            }),
            'cronograma_inicio': forms.DateInput(attrs={
                'class': 'form-control',
                'type': 'date'
            }),
            'cronograma_fim': forms.DateInput(attrs={
                'class': 'form-control',
                'type': 'date'
            }),
            'data_inicio_real': forms.DateInput(attrs={
                'class': 'form-control',
                'type': 'date'
            }),
            'data_fim_real': forms.DateInput(attrs={
                'class': 'form-control',
                'type': 'date'
            }),
            'progresso': forms.NumberInput(attrs={
                'class': 'form-control',
                'min': '0',
                'max': '100',
                'step': '1'
            }),
            'orcamento': forms.NumberInput(attrs={
                'class': 'form-control',
                'step': '0.01',
                'min': '0'
            }),
            'custo_real': forms.NumberInput(attrs={
                'class': 'form-control',
                'step': '0.01',
                'min': '0'
            }),
            'ativo': forms.CheckboxInput(attrs={
                'class': 'form-check-input'
            })
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        
        # Filtrar apenas portfolios ativos
        self.fields['portfolio'].queryset = Portfolio.objects.filter(ativo=True).order_by('nome')
        
        # Adiciona classes CSS aos labels
        for field_name, field in self.fields.items():
            if field_name != 'ativo':  # Não adicionar form-control ao checkbox
                field.widget.attrs.update({'class': field.widget.attrs.get('class', '') + ' form-control'}) 