from django.db import models
from django.core.validators import MinValueValidator, MaxValueValidator
from client.models import Client
from workspace.models import Workspace
from portfolio.models import Portfolio


class Project(models.Model):
    """
    Modelo para representar projetos.
    Um projeto pertence a um portfolio e workspace.
    """
    
    STATUS_CHOICES = [
        ('planejamento', 'Planejamento'),
        ('em_andamento', 'Em Andamento'),
        ('pausado', 'Pausado'),
        ('concluido', 'Concluído'),
        ('cancelado', 'Cancelado'),
    ]
    
    PRIORIDADE_CHOICES = [
        ('baixa', 'Baixa'),
        ('media', 'Média'),
        ('alta', 'Alta'),
        ('urgente', 'Urgente'),
    ]
    
    nome = models.CharField(
        max_length=200,
        verbose_name="Nome",
        help_text="Nome do projeto"
    )
    
    portfolio = models.ForeignKey(
        Portfolio,
        on_delete=models.CASCADE,
        verbose_name="Portfolio",
        help_text="Portfolio ao qual o projeto pertence",
        related_name="projects"
    )
    
    workspace = models.ForeignKey(
        Workspace,
        on_delete=models.CASCADE,
        verbose_name="Workspace",
        help_text="Workspace ao qual o projeto pertence",
        related_name="projects"
    )
    
    cliente = models.ForeignKey(
        Client,
        on_delete=models.CASCADE,
        verbose_name="Cliente",
        help_text="Cliente proprietário do projeto",
        related_name="projects"
    )
    
    descricao = models.TextField(
        blank=True,
        null=True,
        verbose_name="Descrição",
        help_text="Descrição detalhada do projeto"
    )
    
    cronograma_inicio = models.DateField(
        verbose_name="Data de Início",
        help_text="Data de início planejada do projeto"
    )
    
    cronograma_fim = models.DateField(
        verbose_name="Data de Fim",
        help_text="Data de fim planejada do projeto"
    )
    
    data_inicio_real = models.DateField(
        blank=True,
        null=True,
        verbose_name="Data de Início Real",
        help_text="Data real de início do projeto"
    )
    
    data_fim_real = models.DateField(
        blank=True,
        null=True,
        verbose_name="Data de Fim Real",
        help_text="Data real de conclusão do projeto"
    )
    
    progresso = models.IntegerField(
        default=0,
        validators=[MinValueValidator(0), MaxValueValidator(100)],
        verbose_name="Progresso (%)",
        help_text="Percentual de progresso do projeto (0-100)"
    )
    
    status = models.CharField(
        max_length=20,
        choices=STATUS_CHOICES,
        default='planejamento',
        verbose_name="Status",
        help_text="Status atual do projeto"
    )
    
    prioridade = models.CharField(
        max_length=10,
        choices=PRIORIDADE_CHOICES,
        default='media',
        verbose_name="Prioridade",
        help_text="Prioridade do projeto"
    )
    
    orcamento = models.DecimalField(
        max_digits=12,
        decimal_places=2,
        blank=True,
        null=True,
        verbose_name="Orçamento",
        help_text="Orçamento do projeto"
    )
    
    custo_real = models.DecimalField(
        max_digits=12,
        decimal_places=2,
        default=0,
        verbose_name="Custo Real",
        help_text="Custo real do projeto"
    )
    
    ativo = models.BooleanField(
        default=True,
        verbose_name="Ativo",
        help_text="Indica se o projeto está ativo"
    )
    
    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name="Criado em"
    )
    
    updated_at = models.DateTimeField(
        auto_now=True,
        verbose_name="Atualizado em"
    )

    class Meta:
        verbose_name = "Projeto"
        verbose_name_plural = "Projetos"
        ordering = ['portfolio', 'nome']
        unique_together = ['portfolio', 'nome']

    def __str__(self):
        return f"{self.portfolio.nome} - {self.nome}"

    def get_absolute_url(self):
        from django.urls import reverse
        return reverse('project:detail', kwargs={'pk': self.pk})

    @property
    def duracao_planejada(self):
        """Retorna a duração planejada em dias."""
        if self.cronograma_inicio and self.cronograma_fim:
            return (self.cronograma_fim - self.cronograma_inicio).days
        return 0

    @property
    def duracao_real(self):
        """Retorna a duração real em dias."""
        if self.data_inicio_real and self.data_fim_real:
            return (self.data_fim_real - self.data_inicio_real).days
        return 0

    @property
    def total_boards(self):
        return self.boards.count()

    @property
    def total_tasks(self):
        total = 0
        for board in self.boards.all():
            for group in board.groups.all():
                total += group.tasks.count()
        return total

    @property
    def tasks_concluidas(self):
        total = 0
        for board in self.boards.all():
            for group in board.groups.all():
                total += group.tasks.filter(status='concluida').count()
        return total

    @property
    def is_atrasado(self):
        """Verifica se o projeto está atrasado."""
        from django.utils import timezone
        if self.cronograma_fim and self.status not in ['concluido', 'cancelado']:
            return timezone.now().date() > self.cronograma_fim
        return False

    def get_status_color(self):
        """Retorna a cor do badge baseada no status."""
        colors = {
            'planejamento': 'secondary',
            'em_andamento': 'primary',
            'pausado': 'warning',
            'concluido': 'success',
            'cancelado': 'danger',
        }
        return colors.get(self.status, 'secondary')

    def get_prioridade_color(self):
        """Retorna a cor do badge baseada na prioridade."""
        colors = {
            'baixa': 'success',
            'media': 'info',
            'alta': 'warning',
            'urgente': 'danger',
        }
        return colors.get(self.prioridade, 'secondary')

    def save(self, *args, **kwargs):
        # Garante que workspace e cliente sejam consistentes com o portfolio
        if self.portfolio:
            self.workspace = self.portfolio.workspace
            self.cliente = self.portfolio.cliente
        super().save(*args, **kwargs)

