from rest_framework import serializers
from .models import Project
from portfolio.serializers import PortfolioListSerializer
from workspace.serializers import WorkspaceListSerializer
from client.serializers import ClientListSerializer


class ProjectSerializer(serializers.ModelSerializer):
    """
    Serializer para o modelo Project.
    """
    
    portfolio_nome = serializers.CharField(source='portfolio.nome', read_only=True)
    workspace_nome = serializers.CharField(source='workspace.nome', read_only=True)
    cliente_nome = serializers.CharField(source='cliente.nome', read_only=True)
    duracao_planejada = serializers.ReadOnlyField()
    duracao_real = serializers.ReadOnlyField()
    total_boards = serializers.ReadOnlyField()
    total_tasks = serializers.ReadOnlyField()
    tasks_concluidas = serializers.ReadOnlyField()
    is_atrasado = serializers.ReadOnlyField()
    
    class Meta:
        model = Project
        fields = [
            'id', 'nome', 'portfolio', 'portfolio_nome', 'workspace', 'workspace_nome',
            'cliente', 'cliente_nome', 'descricao', 'cronograma_inicio', 'cronograma_fim',
            'data_inicio_real', 'data_fim_real', 'progresso', 'status', 'prioridade',
            'orcamento', 'custo_real', 'ativo',
            'duracao_planejada', 'duracao_real', 'total_boards', 'total_tasks',
            'tasks_concluidas', 'is_atrasado',
            'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'workspace', 'cliente', 'created_at', 'updated_at']


class ProjectDetailSerializer(serializers.ModelSerializer):
    """
    Serializer detalhado para o modelo Project.
    """
    
    portfolio = PortfolioListSerializer(read_only=True)
    workspace = WorkspaceListSerializer(read_only=True)
    cliente = ClientListSerializer(read_only=True)
    boards = serializers.SerializerMethodField()
    duracao_planejada = serializers.ReadOnlyField()
    duracao_real = serializers.ReadOnlyField()
    is_atrasado = serializers.ReadOnlyField()
    
    class Meta:
        model = Project
        fields = [
            'id', 'nome', 'portfolio', 'workspace', 'cliente', 'descricao',
            'cronograma_inicio', 'cronograma_fim', 'data_inicio_real', 'data_fim_real',
            'progresso', 'status', 'prioridade', 'orcamento', 'custo_real', 'ativo',
            'boards', 'duracao_planejada', 'duracao_real', 'is_atrasado',
            'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'workspace', 'cliente', 'created_at', 'updated_at']
    
    def get_boards(self, obj):
        from board.serializers import BoardListSerializer
        boards = obj.boards.filter(ativo=True)
        return BoardListSerializer(boards, many=True).data


class ProjectListSerializer(serializers.ModelSerializer):
    """
    Serializer simplificado para listagem de projetos.
    """
    
    portfolio_nome = serializers.CharField(source='portfolio.nome', read_only=True)
    workspace_nome = serializers.CharField(source='workspace.nome', read_only=True)
    cliente_nome = serializers.CharField(source='cliente.nome', read_only=True)
    total_tasks = serializers.ReadOnlyField()
    tasks_concluidas = serializers.ReadOnlyField()
    is_atrasado = serializers.ReadOnlyField()
    
    class Meta:
        model = Project
        fields = [
            'id', 'nome', 'portfolio_nome', 'workspace_nome', 'cliente_nome',
            'cronograma_inicio', 'cronograma_fim', 'progresso', 'status', 'prioridade',
            'total_tasks', 'tasks_concluidas', 'is_atrasado', 'ativo'
        ]


class ProjectCreateSerializer(serializers.ModelSerializer):
    """
    Serializer para criação de projetos.
    """
    
    class Meta:
        model = Project
        fields = [
            'nome', 'portfolio', 'descricao', 'cronograma_inicio', 'cronograma_fim',
            'progresso', 'status', 'prioridade', 'orcamento'
        ]
    
    def create(self, validated_data):
        # O workspace e cliente são automaticamente definidos pelo portfolio
        portfolio = validated_data['portfolio']
        validated_data['workspace'] = portfolio.workspace
        validated_data['cliente'] = portfolio.cliente
        return super().create(validated_data)


class ProjectUpdateStatusSerializer(serializers.ModelSerializer):
    """
    Serializer para atualização de status do projeto.
    """
    
    class Meta:
        model = Project
        fields = ['status', 'progresso', 'data_inicio_real', 'data_fim_real', 'custo_real']

