from django.shortcuts import render, get_object_or_404, redirect
from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.core.paginator import Paginator
from django.db.models import Q, Count
from django.http import JsonResponse
from django.views.generic import CreateView, UpdateView
from django.contrib.auth.mixins import LoginRequiredMixin
from django.urls import reverse_lazy
from .models import Project
from .forms import ProjectForm, ProjectCreateForm
from portfolio.models import Portfolio
from workspace.models import Workspace
from client.models import Client
from permissions.models import VisibilityPermission
from gestao_agencia.decorators import superuser_required
from gestao_agencia.mixins import SuperuserRequiredMixin


@login_required
def project_list(request):
    """
    Lista todos os projetos com filtros e paginação.
    """
    # Obter parâmetros de filtro
    search = request.GET.get('search', '')
    status = request.GET.get('status', '')
    workspace_id = request.GET.get('workspace', '')
    portfolio_id = request.GET.get('portfolio', '')
    
    # Query base - filtra apenas projetos acessíveis
    projects = VisibilityPermission.get_accessible_objects(request.user, Project)
    projects = projects.select_related('portfolio', 'workspace', 'cliente')
    
    # Aplicar filtros
    if search:
        projects = projects.filter(
            Q(nome__icontains=search) |
            Q(portfolio__nome__icontains=search) |
            Q(workspace__nome__icontains=search) |
            Q(cliente__nome__icontains=search) |
            Q(descricao__icontains=search)
        )
    
    if status:
        projects = projects.filter(status=status)
    
    if workspace_id:
        projects = projects.filter(workspace_id=workspace_id)
    
    if portfolio_id:
        projects = projects.filter(portfolio_id=portfolio_id)
    
    # Ordenar por portfolio e nome
    projects = projects.order_by('portfolio__nome', 'nome')
    
    # Paginação
    paginator = Paginator(projects, 12)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)
    
    # Estatísticas baseadas nos projetos acessíveis
    accessible_projects = VisibilityPermission.get_accessible_objects(request.user, Project)
    total_projects = accessible_projects.count()
    projects_concluidos = accessible_projects.filter(status='concluido').count()
    projects_em_andamento = accessible_projects.filter(status='em_andamento').count()
    projects_pausados = accessible_projects.filter(status='pausado').count()
    
    # Lista de workspaces e portfolios acessíveis para o filtro
    accessible_workspaces = VisibilityPermission.get_accessible_objects(request.user, Workspace)
    accessible_portfolios = VisibilityPermission.get_accessible_objects(request.user, Portfolio)
    workspaces = accessible_workspaces.order_by('nome')
    portfolios = accessible_portfolios.order_by('workspace__nome', 'nome')
    
    context = {
        'projects': page_obj,
        'page_obj': page_obj,
        'is_paginated': page_obj.has_other_pages(),
        'search': search,
        'status': status,
        'workspace_id': workspace_id,
        'portfolio_id': portfolio_id,
        'total_projects': total_projects,
        'projects_concluidos': projects_concluidos,
        'projects_em_andamento': projects_em_andamento,
        'projects_pausados': projects_pausados,
        'workspaces': workspaces,
        'portfolios': portfolios,
    }
    
    return render(request, 'project/list.html', context)


@login_required
@superuser_required
def project_create(request):
    """
    Cria um novo projeto.
    """
    if request.method == 'POST':
        # Implementação básica - será expandida com forms
        nome = request.POST.get('nome')
        portfolio_id = request.POST.get('portfolio')
        descricao = request.POST.get('descricao', '')
        cronograma_inicio = request.POST.get('cronograma_inicio')
        cronograma_fim = request.POST.get('cronograma_fim')
        
        if nome and portfolio_id and cronograma_inicio and cronograma_fim:
            try:
                portfolio = Portfolio.objects.get(id=portfolio_id)
                project = Project.objects.create(
                    nome=nome,
                    portfolio=portfolio,
                    workspace=portfolio.workspace,
                    cliente=portfolio.cliente,
                    descricao=descricao,
                    cronograma_inicio=cronograma_inicio,
                    cronograma_fim=cronograma_fim
                )
                messages.success(request, f'Projeto "{project.nome}" criado com sucesso!')
                return redirect('project:detail', pk=project.pk)
            except Portfolio.DoesNotExist:
                messages.error(request, 'Portfolio não encontrado.')
        else:
            messages.error(request, 'Nome, portfolio e datas são obrigatórios.')
    
    # GET request - mostrar formulário
    portfolios = Portfolio.objects.filter(ativo=True).order_by('workspace__nome', 'nome')
    
    context = {
        'portfolios': portfolios,
        'project': None,
    }
    
    return render(request, 'project/form.html', context)


@login_required
def project_detail(request, pk):
    """
    Exibe os detalhes de um projeto.
    """
    project = get_object_or_404(Project, pk=pk)
    
    # Verifica se o usuário tem acesso ao projeto
    if not VisibilityPermission.has_access(request.user, project):
        messages.error(request, "Acesso negado. Você não tem permissão para visualizar este projeto.")
        return redirect('project:list')
    
    context = {
        'project': project,
    }
    
    return render(request, 'project/detail.html', context)


@login_required
@superuser_required
def project_edit(request, pk):
    """
    Edita um projeto existente.
    """
    project = get_object_or_404(Project, pk=pk)
    
    if request.method == 'POST':
        # Implementação básica - será expandida com forms
        nome = request.POST.get('nome')
        descricao = request.POST.get('descricao', '')
        status = request.POST.get('status', project.status)
        cronograma_inicio = request.POST.get('cronograma_inicio')
        cronograma_fim = request.POST.get('cronograma_fim')
        
        if nome and cronograma_inicio and cronograma_fim:
            project.nome = nome
            project.descricao = descricao
            project.status = status
            project.cronograma_inicio = cronograma_inicio
            project.cronograma_fim = cronograma_fim
            project.save()
            messages.success(request, f'Projeto "{project.nome}" atualizado com sucesso!')
            return redirect('project:detail', pk=project.pk)
        else:
            messages.error(request, 'Nome e datas são obrigatórios.')
    
    # GET request - mostrar formulário
    portfolios = Portfolio.objects.filter(ativo=True).order_by('workspace__nome', 'nome')
    
    context = {
        'project': project,
        'portfolios': portfolios,
    }
    
    return render(request, 'project/form.html', context)


@login_required
@superuser_required
def project_delete(request, pk):
    """
    Exclui um projeto.
    """
    project = get_object_or_404(Project, pk=pk)
    
    if request.method == 'POST':
        nome = project.nome
        project.delete()
        messages.success(request, f'Projeto "{nome}" excluído com sucesso!')
        return redirect('project:list')
    
    context = {
        'project': project,
    }
    
    return render(request, 'project/delete.html', context)


class ProjectCreateModalView(LoginRequiredMixin, SuperuserRequiredMixin, CreateView):
    """
    View para criação de project via modal.
    """
    model = Project
    form_class = ProjectCreateForm
    template_name = 'project/modal_create.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context.update({
            'modal_id': 'createProjectModal',
            'modal_title': 'Novo Projeto',
            'modal_icon': 'bi-kanban',
            'form_id': 'projectForm',
            'submit_text': 'Criar Projeto'
        })
        return context
    
    def form_valid(self, form):
        try:
            project = form.save(commit=False)
            # Garante que workspace e cliente sejam definidos baseado no portfolio
            if project.portfolio:
                project.workspace = project.portfolio.workspace
                project.cliente = project.portfolio.cliente
            # Define valores padrão para campos automáticos
            project.progresso = 0
            project.custo_real = 0
            project.save()
            
            if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
                return JsonResponse({
                    'success': True,
                    'message': 'Projeto criado com sucesso!',
                    'obj': {
                        'id': project.id,
                        'nome': project.nome,
                        'detail_url': reverse_lazy('project:detail', kwargs={'pk': project.id})
                    }
                })
            return super().form_valid(form)
        except Exception as e:
            print(f"Erro ao criar projeto: {e}")
            if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
                return JsonResponse({
                    'success': False,
                    'errors': {'__all__': [str(e)]}
                })
            raise
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)


class ProjectUpdateModalView(LoginRequiredMixin, SuperuserRequiredMixin, UpdateView):
    """
    View para edição de project via modal.
    """
    model = Project
    form_class = ProjectCreateForm
    template_name = 'project/modal_edit.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        project = self.get_object()
        context.update({
            'modal_id': 'editProjectModal',
            'modal_title': f'Editar Projeto: {project.nome}',
            'modal_icon': 'bi-kanban-gear',
            'form_id': 'projectEditForm',
            'submit_text': 'Atualizar Projeto'
        })
        return context
    
    def form_valid(self, form):
        try:
            project = form.save(commit=False)
            # Garante que workspace e cliente sejam definidos baseado no portfolio
            if project.portfolio:
                project.workspace = project.portfolio.workspace
                project.cliente = project.portfolio.cliente
            project.save()
            
            if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
                return JsonResponse({
                    'success': True,
                    'message': 'Projeto atualizado com sucesso!',
                    'obj': {
                        'id': project.id,
                        'nome': project.nome,
                        'status': project.status,
                        'ativo': project.ativo,
                        'detail_url': reverse_lazy('project:detail', kwargs={'pk': project.id})
                    }
                })
            return super().form_valid(form)
        except Exception as e:
            print(f"Erro ao atualizar projeto: {e}")
            if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
                return JsonResponse({
                    'success': False,
                    'errors': {'__all__': [str(e)]}
                })
            raise
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)
