#!/usr/bin/env python3
"""
Script para executar todos os testes do projeto de gestão de agência.
Este script executa testes unitários, de integração e de API para validar
se todo o sistema está funcionando corretamente.
"""

import os
import sys
import subprocess
import time
from datetime import datetime
import json

# Configurações
DJANGO_SETTINGS_MODULE = 'gestao_agencia.settings'
TEST_APPS = [
    'task',
    'client', 
    'collaborator',
    'project',
    'board',
    'portfolio',
    'workspace',
    'notification'
]

class TestRunner:
    """Classe para executar e gerenciar testes"""
    
    def __init__(self):
        self.results = {}
        self.start_time = None
        self.end_time = None
        
    def run_command(self, command):
        """Executa um comando e retorna o resultado"""
        try:
            result = subprocess.run(
                command,
                shell=True,
                capture_output=True,
                text=True,
                timeout=300  # 5 minutos de timeout
            )
            return {
                'success': result.returncode == 0,
                'stdout': result.stdout,
                'stderr': result.stderr,
                'returncode': result.returncode
            }
        except subprocess.TimeoutExpired:
            return {
                'success': False,
                'stdout': '',
                'stderr': 'Timeout: comando demorou mais de 5 minutos',
                'returncode': -1
            }
        except Exception as e:
            return {
                'success': False,
                'stdout': '',
                'stderr': str(e),
                'returncode': -1
            }
    
    def check_django_environment(self):
        """Verifica se o ambiente Django está configurado corretamente"""
        print("🔍 Verificando ambiente Django...")
        
        # Verificar se manage.py existe
        if not os.path.exists('manage.py'):
            print("❌ Erro: manage.py não encontrado. Execute este script na raiz do projeto.")
            return False
        
        # Verificar se as configurações Django estão disponíveis
        try:
            os.environ.setdefault('DJANGO_SETTINGS_MODULE', DJANGO_SETTINGS_MODULE)
            import django
            django.setup()
            print("✅ Ambiente Django configurado com sucesso")
            return True
        except Exception as e:
            print(f"❌ Erro ao configurar Django: {e}")
            return False
    
    def run_migrations(self):
        """Executa as migrações do banco de dados"""
        print("🔄 Executando migrações...")
        
        result = self.run_command('python manage.py migrate --verbosity=0')
        if result['success']:
            print("✅ Migrações executadas com sucesso")
        else:
            print(f"❌ Erro nas migrações: {result['stderr']}")
        
        return result['success']
    
    def run_tests_for_app(self, app_name):
        """Executa testes para um app específico"""
        print(f"\n🧪 Executando testes para {app_name}...")
        
        command = f'python manage.py test {app_name} --verbosity=2 --noinput'
        result = self.run_command(command)
        
        self.results[app_name] = result
        
        if result['success']:
            print(f"✅ Testes de {app_name} passaram")
        else:
            print(f"❌ Testes de {app_name} falharam")
            if result['stderr']:
                print(f"   Erro: {result['stderr']}")
        
        return result['success']
    
    def run_all_tests(self):
        """Executa todos os testes"""
        print("🚀 Iniciando execução de todos os testes...")
        self.start_time = datetime.now()
        
        all_passed = True
        
        for app in TEST_APPS:
            if not self.run_tests_for_app(app):
                all_passed = False
        
        self.end_time = datetime.now()
        
        return all_passed
    
    def run_coverage_tests(self):
        """Executa testes com cobertura de código"""
        print("\n📊 Executando testes com cobertura...")
        
        # Verificar se coverage está instalado
        try:
            import coverage
        except ImportError:
            print("⚠️  Coverage não instalado. Instalando...")
            self.run_command('pip install coverage')
        
        # Executar testes com cobertura
        commands = [
            'coverage run --source="." manage.py test --verbosity=1',
            'coverage report --show-missing',
            'coverage html --directory=htmlcov'
        ]
        
        for command in commands:
            result = self.run_command(command)
            if not result['success']:
                print(f"❌ Erro no comando: {command}")
                print(f"   Erro: {result['stderr']}")
        
        print("✅ Relatório de cobertura gerado em htmlcov/")
    
    def run_system_checks(self):
        """Executa verificações do sistema Django"""
        print("\n🔧 Executando verificações do sistema...")
        
        commands = [
            'python manage.py check --deploy',
            'python manage.py check --database default',
            'python manage.py check --tags security'
        ]
        
        all_checks_passed = True
        
        for command in commands:
            result = self.run_command(command)
            if result['success']:
                print(f"✅ {command.split()[-1]} passou")
            else:
                print(f"❌ {command.split()[-1]} falhou")
                print(f"   Erro: {result['stderr']}")
                all_checks_passed = False
        
        return all_checks_passed
    
    def generate_report(self):
        """Gera relatório dos testes"""
        print("\n📋 Gerando relatório...")
        
        duration = self.end_time - self.start_time if self.end_time else None
        
        report = {
            'timestamp': datetime.now().isoformat(),
            'duration': str(duration) if duration else None,
            'summary': {
                'total_apps': len(TEST_APPS),
                'passed': sum(1 for r in self.results.values() if r['success']),
                'failed': sum(1 for r in self.results.values() if not r['success'])
            },
            'results': {}
        }
        
        for app, result in self.results.items():
            report['results'][app] = {
                'success': result['success'],
                'returncode': result['returncode'],
                'has_errors': bool(result['stderr']),
                'error_message': result['stderr'] if result['stderr'] else None
            }
        
        # Salvar relatório em JSON
        with open('test_report.json', 'w', encoding='utf-8') as f:
            json.dump(report, f, indent=2, ensure_ascii=False)
        
        # Gerar relatório em texto
        with open('test_report.txt', 'w', encoding='utf-8') as f:
            f.write("RELATÓRIO DE TESTES - SISTEMA DE GESTÃO DE AGÊNCIA\n")
            f.write("=" * 60 + "\n\n")
            f.write(f"Data/Hora: {report['timestamp']}\n")
            f.write(f"Duração: {report['duration']}\n\n")
            
            f.write("RESUMO:\n")
            f.write(f"- Total de apps testados: {report['summary']['total_apps']}\n")
            f.write(f"- Apps com sucesso: {report['summary']['passed']}\n")
            f.write(f"- Apps com falha: {report['summary']['failed']}\n\n")
            
            f.write("DETALHES POR APP:\n")
            f.write("-" * 40 + "\n")
            
            for app, result in report['results'].items():
                status = "✅ PASSOU" if result['success'] else "❌ FALHOU"
                f.write(f"{app}: {status}\n")
                if result['error_message']:
                    f.write(f"  Erro: {result['error_message']}\n")
                f.write("\n")
        
        print("✅ Relatórios gerados:")
        print("   - test_report.json (formato JSON)")
        print("   - test_report.txt (formato texto)")
    
    def print_summary(self):
        """Imprime resumo dos resultados"""
        print("\n" + "=" * 60)
        print("RESUMO DOS TESTES")
        print("=" * 60)
        
        if self.end_time and self.start_time:
            duration = self.end_time - self.start_time
            print(f"⏱️  Duração total: {duration}")
        
        total_apps = len(self.results)
        passed = sum(1 for r in self.results.values() if r['success'])
        failed = total_apps - passed
        
        print(f"📊 Apps testados: {total_apps}")
        print(f"✅ Aprovados: {passed}")
        print(f"❌ Reprovados: {failed}")
        
        if failed > 0:
            print("\n❌ Apps com falha:")
            for app, result in self.results.items():
                if not result['success']:
                    print(f"   - {app}")
        
        print("\n" + "=" * 60)
        
        if failed == 0:
            print("🎉 TODOS OS TESTES PASSARAM!")
        else:
            print(f"⚠️  {failed} app(s) com falha. Verifique os relatórios para mais detalhes.")
    
    def run(self):
        """Executa todo o processo de testes"""
        print("🧪 EXECUTOR DE TESTES - SISTEMA DE GESTÃO DE AGÊNCIA")
        print("=" * 60)
        
        # Verificar ambiente
        if not self.check_django_environment():
            return False
        
        # Executar migrações
        if not self.run_migrations():
            print("❌ Falha nas migrações. Abortando testes.")
            return False
        
        # Executar verificações do sistema
        self.run_system_checks()
        
        # Executar todos os testes
        all_passed = self.run_all_tests()
        
        # Executar testes com cobertura
        self.run_coverage_tests()
        
        # Gerar relatórios
        self.generate_report()
        
        # Imprimir resumo
        self.print_summary()
        
        return all_passed


def main():
    """Função principal"""
    runner = TestRunner()
    success = runner.run()
    
    if success:
        print("\n🎯 Todos os testes foram executados com sucesso!")
        sys.exit(0)
    else:
        print("\n💥 Alguns testes falharam. Verifique os relatórios.")
        sys.exit(1)


if __name__ == '__main__':
    main() 