#!/usr/bin/env python3
"""
Script de inicialização para o projeto Gestão de Agência.
Este script configura o banco de dados, cria superusuário e carrega dados de exemplo.
"""

import os
import sys
import django
from django.core.management import execute_from_command_line

def setup_django():
    """Configura o Django."""
    os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'gestao_agencia.settings')
    django.setup()

def run_command(command):
    """Executa um comando Django."""
    print(f"Executando: {' '.join(command)}")
    try:
        execute_from_command_line(command)
        print("✓ Comando executado com sucesso\n")
        return True
    except Exception as e:
        print(f"✗ Erro ao executar comando: {e}\n")
        return False

def create_superuser():
    """Cria superusuário se não existir."""
    from django.contrib.auth.models import User
    
    if not User.objects.filter(is_superuser=True).exists():
        print("Criando superusuário...")
        User.objects.create_superuser(
            username='admin',
            email='admin@gestaoagencia.com',
            password='admin123'
        )
        print("✓ Superusuário criado (admin/admin123)\n")
    else:
        print("✓ Superusuário já existe\n")

def main():
    """Função principal."""
    print("=" * 60)
    print("CONFIGURAÇÃO INICIAL - GESTÃO DE AGÊNCIA")
    print("=" * 60)
    print()
    
    # Configurar Django
    setup_django()
    
    # 1. Fazer migrações
    print("1. Criando migrações...")
    if not run_command(['manage.py', 'makemigrations']):
        return False
    
    print("2. Aplicando migrações...")
    if not run_command(['manage.py', 'migrate']):
        return False
    
    # 3. Criar superusuário
    print("3. Configurando superusuário...")
    create_superuser()
    
    # 4. Configurar grupos e permissões
    print("4. Configurando grupos e permissões...")
    if not run_command(['manage.py', 'setup_groups']):
        return False
    
    # 5. Carregar dados de exemplo
    print("5. Carregando dados de exemplo...")
    if not run_command(['manage.py', 'load_sample_data']):
        return False
    
    # 6. Coletar arquivos estáticos
    print("6. Coletando arquivos estáticos...")
    if not run_command(['manage.py', 'collectstatic', '--noinput']):
        print("⚠ Aviso: Erro ao coletar arquivos estáticos (normal em desenvolvimento)\n")
    
    print("=" * 60)
    print("CONFIGURAÇÃO CONCLUÍDA COM SUCESSO!")
    print("=" * 60)
    print()
    print("Credenciais de acesso:")
    print("- Admin: admin / admin123")
    print("- Gerente: gerente / 123456")
    print("- Colaboradores: ana.dev, carlos.design, etc. / 123456")
    print()
    print("Para iniciar o servidor:")
    print("python manage.py runserver")
    print()
    print("Para iniciar o Celery (opcional):")
    print("celery -A gestao_agencia worker --loglevel=info")
    print("celery -A gestao_agencia beat --loglevel=info")
    print()

if __name__ == '__main__':
    main()

