// Custom JavaScript for Gestão de Projetos

$(document).ready(function() {
    // Initialize tooltips
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });

    // Initialize popovers
    var popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
    var popoverList = popoverTriggerList.map(function (popoverTriggerEl) {
        return new bootstrap.Popover(popoverTriggerEl);
    });

    // Load notifications
    loadNotifications();

    // Auto-refresh notifications every 30 seconds
    setInterval(loadNotifications, 30000);

    // Time control buttons
    $('.time-control-btn').on('click', function(e) {
        e.preventDefault();
        var taskId = $(this).data('task-id');
        var action = $(this).data('action');
        
        timeControl(taskId, action, $(this));
    });

    // Status update buttons
    $('.status-update-btn').on('click', function(e) {
        e.preventDefault();
        var taskId = $(this).data('task-id');
        var status = $(this).data('status');
        
        updateTaskStatus(taskId, status);
    });

    // Progress update
    $('.progress-update').on('change', function() {
        var taskId = $(this).data('task-id');
        var progress = $(this).val();
        
        updateProgress(taskId, progress);
    });

    // Confirm delete actions
    $('.delete-btn').on('click', function(e) {
        e.preventDefault();
        var url = $(this).attr('href');
        var itemName = $(this).data('item-name') || 'este item';
        
        if (confirm('Tem certeza que deseja excluir ' + itemName + '?')) {
            window.location.href = url;
        }
    });

    // Auto-save forms
    $('.auto-save').on('change', function() {
        var form = $(this).closest('form');
        autoSaveForm(form);
    });

    // Search functionality
    $('#search-input').on('keyup', function() {
        var searchTerm = $(this).val().toLowerCase();
        filterTable(searchTerm);
    });

    // Kanban drag and drop (if sortable is available)
    if (typeof Sortable !== 'undefined') {
        initializeKanban();
    }
});

// Load notifications
function loadNotifications() {
    $.ajax({
        url: '/api/notifications/',
        method: 'GET',
        data: { lida: false, limit: 5 },
        success: function(data) {
            updateNotificationDropdown(data.results);
            updateNotificationCount(data.count);
        },
        error: function() {
            console.log('Erro ao carregar notificações');
        }
    });
}

// Update notification dropdown
function updateNotificationDropdown(notifications) {
    var notificationsList = $('#notifications-list');
    notificationsList.empty();

    if (notifications.length === 0) {
        notificationsList.append('<li><span class="dropdown-item-text text-muted">Nenhuma notificação</span></li>');
    } else {
        notifications.forEach(function(notification) {
            var item = $('<li><a class="dropdown-item notification-item" href="#"></a></li>');
            var link = item.find('a');
            
            link.html(
                '<div class="notification-title">' + notification.titulo + '</div>' +
                '<div class="notification-time">' + formatDate(notification.created_at) + '</div>'
            );
            
            link.on('click', function(e) {
                e.preventDefault();
                markNotificationAsRead(notification.id);
            });
            
            notificationsList.append(item);
        });
    }
}

// Update notification count
function updateNotificationCount(count) {
    var badge = $('#notification-count');
    badge.text(count);
    
    if (count > 0) {
        badge.show();
    } else {
        badge.hide();
    }
}

// Mark notification as read
function markNotificationAsRead(notificationId) {
    $.ajax({
        url: '/api/notifications/' + notificationId + '/mark_as_read/',
        method: 'PATCH',
        headers: {
            'X-CSRFToken': getCookie('csrftoken')
        },
        success: function() {
            loadNotifications();
        }
    });
}

// Time control for tasks
function timeControl(taskId, action, button) {
    showSpinner();
    
    $.ajax({
        url: '/api/tasks/' + taskId + '/time_control/',
        method: 'POST',
        headers: {
            'X-CSRFToken': getCookie('csrftoken')
        },
        data: {
            action: action
        },
        success: function(data) {
            hideSpinner();
            updateTimeControlButtons(taskId, data.status);
            showToast('Ação realizada com sucesso!', 'success');
            
            // Reload page to update status
            setTimeout(function() {
                location.reload();
            }, 1000);
        },
        error: function(xhr) {
            hideSpinner();
            var error = xhr.responseJSON ? xhr.responseJSON.error : 'Erro ao executar ação';
            showToast(error, 'error');
        }
    });
}

// Update task status
function updateTaskStatus(taskId, status) {
    showSpinner();
    
    $.ajax({
        url: '/api/tasks/' + taskId + '/update_status/',
        method: 'PATCH',
        headers: {
            'X-CSRFToken': getCookie('csrftoken')
        },
        data: {
            status: status
        },
        success: function(data) {
            hideSpinner();
            showToast('Status atualizado com sucesso!', 'success');
            location.reload();
        },
        error: function(xhr) {
            hideSpinner();
            var error = xhr.responseJSON ? xhr.responseJSON.error : 'Erro ao atualizar status';
            showToast(error, 'error');
        }
    });
}

// Update progress
function updateProgress(taskId, progress) {
    $.ajax({
        url: '/api/tasks/' + taskId + '/',
        method: 'PATCH',
        headers: {
            'X-CSRFToken': getCookie('csrftoken')
        },
        data: {
            esforco_utilizado: progress
        },
        success: function(data) {
            showToast('Progresso atualizado!', 'success');
        },
        error: function() {
            showToast('Erro ao atualizar progresso', 'error');
        }
    });
}

// Auto-save form
function autoSaveForm(form) {
    var formData = form.serialize();
    var url = form.attr('action');
    
    $.ajax({
        url: url,
        method: 'POST',
        data: formData,
        success: function() {
            showToast('Salvo automaticamente', 'info');
        }
    });
}

// Filter table
function filterTable(searchTerm) {
    $('.table tbody tr').each(function() {
        var row = $(this);
        var text = row.text().toLowerCase();
        
        if (text.indexOf(searchTerm) === -1) {
            row.hide();
        } else {
            row.show();
        }
    });
}

// Initialize Kanban board
function initializeKanban() {
    $('.kanban-column').each(function() {
        var column = this;
        
        Sortable.create(column.querySelector('.kanban-tasks'), {
            group: 'kanban',
            animation: 150,
            onEnd: function(evt) {
                var taskId = evt.item.dataset.taskId;
                var newGroupId = evt.to.dataset.groupId;
                
                updateTaskGroup(taskId, newGroupId);
            }
        });
    });
}

// Update task group (for Kanban)
function updateTaskGroup(taskId, groupId) {
    $.ajax({
        url: '/api/tasks/' + taskId + '/',
        method: 'PATCH',
        headers: {
            'X-CSRFToken': getCookie('csrftoken')
        },
        data: {
            group: groupId
        },
        success: function() {
            showToast('Tarefa movida com sucesso!', 'success');
        },
        error: function() {
            showToast('Erro ao mover tarefa', 'error');
        }
    });
}

// Show spinner
function showSpinner() {
    var spinner = $('<div class="spinner-overlay"><div class="spinner-border text-primary" role="status"></div></div>');
    $('body').append(spinner);
}

// Hide spinner
function hideSpinner() {
    $('.spinner-overlay').remove();
}

// Show toast notification
function showToast(message, type) {
    var alertClass = 'alert-info';
    
    switch(type) {
        case 'success':
            alertClass = 'alert-success';
            break;
        case 'error':
            alertClass = 'alert-danger';
            break;
        case 'warning':
            alertClass = 'alert-warning';
            break;
    }
    
    var toast = $('<div class="alert ' + alertClass + ' alert-dismissible fade show position-fixed" style="top: 20px; right: 20px; z-index: 9999;">' +
                  message +
                  '<button type="button" class="btn-close" data-bs-dismiss="alert"></button>' +
                  '</div>');
    
    $('body').append(toast);
    
    setTimeout(function() {
        toast.alert('close');
    }, 3000);
}

// Format date
function formatDate(dateString) {
    var date = new Date(dateString);
    var now = new Date();
    var diff = now - date;
    
    var minutes = Math.floor(diff / 60000);
    var hours = Math.floor(diff / 3600000);
    var days = Math.floor(diff / 86400000);
    
    if (minutes < 1) {
        return 'Agora';
    } else if (minutes < 60) {
        return minutes + ' min atrás';
    } else if (hours < 24) {
        return hours + ' h atrás';
    } else if (days < 7) {
        return days + ' dias atrás';
    } else {
        return date.toLocaleDateString('pt-BR');
    }
}

// Get CSRF token
function getCookie(name) {
    var cookieValue = null;
    if (document.cookie && document.cookie !== '') {
        var cookies = document.cookie.split(';');
        for (var i = 0; i < cookies.length; i++) {
            var cookie = cookies[i].trim();
            if (cookie.substring(0, name.length + 1) === (name + '=')) {
                cookieValue = decodeURIComponent(cookie.substring(name.length + 1));
                break;
            }
        }
    }
    return cookieValue;
}

// Update time control buttons based on task status
function updateTimeControlButtons(taskId, status) {
    var container = $('.time-controls[data-task-id="' + taskId + '"]');
    
    container.find('.time-control-btn').hide();
    
    switch(status) {
        case 'nao_iniciada':
        case 'pausada':
            container.find('[data-action="iniciar"]').show();
            break;
        case 'em_andamento':
            container.find('[data-action="pausar"]').show();
            container.find('[data-action="concluir"]').show();
            break;
        case 'concluida':
        case 'cancelada':
            // No buttons for completed/cancelled tasks
            break;
    }
}

