/**
 * Controle de Tempo para Tarefas
 * Funcionalidades para iniciar, pausar, retomar e parar o trabalho em tarefas
 */

// Variáveis globais para controle de tempo
let cronometros = {};

// Funções principais de controle de tempo
function iniciarTarefa(taskId) {
    if (confirm('Deseja iniciar o trabalho nesta tarefa?')) {
        fetch(`/task/${taskId}/iniciar/`, {
            method: 'POST',
            headers: {
                'X-CSRFToken': getCookie('csrftoken'),
                'Content-Type': 'application/json',
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                mostrarMensagem(data.message, 'success');
                atualizarInterfaceTempo(taskId, data);
                iniciarCronometro(taskId);
            } else {
                mostrarMensagem(data.message, 'error');
            }
        })
        .catch(error => {
            console.error('Erro:', error);
            mostrarMensagem('Erro ao iniciar tarefa', 'error');
        });
    }
}

function pausarTarefa(taskId) {
    if (confirm('Deseja pausar o trabalho nesta tarefa?')) {
        fetch(`/task/${taskId}/pausar/`, {
            method: 'POST',
            headers: {
                'X-CSRFToken': getCookie('csrftoken'),
                'Content-Type': 'application/json',
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                mostrarMensagem(data.message, 'success');
                atualizarInterfaceTempo(taskId, data);
                pararCronometro(taskId);
            } else {
                mostrarMensagem(data.message, 'error');
            }
        })
        .catch(error => {
            console.error('Erro:', error);
            mostrarMensagem('Erro ao pausar tarefa', 'error');
        });
    }
}

function retomarTarefa(taskId) {
    if (confirm('Deseja retomar o trabalho nesta tarefa?')) {
        fetch(`/task/${taskId}/retomar/`, {
            method: 'POST',
            headers: {
                'X-CSRFToken': getCookie('csrftoken'),
                'Content-Type': 'application/json',
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                mostrarMensagem(data.message, 'success');
                atualizarInterfaceTempo(taskId, data);
                iniciarCronometro(taskId);
            } else {
                mostrarMensagem(data.message, 'error');
            }
        })
        .catch(error => {
            console.error('Erro:', error);
            mostrarMensagem('Erro ao retomar tarefa', 'error');
        });
    }
}

function pararTarefa(taskId) {
    if (confirm('Deseja parar completamente o trabalho nesta tarefa?')) {
        fetch(`/task/${taskId}/parar/`, {
            method: 'POST',
            headers: {
                'X-CSRFToken': getCookie('csrftoken'),
                'Content-Type': 'application/json',
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                mostrarMensagem(data.message, 'success');
                atualizarInterfaceTempo(taskId, data);
                pararCronometro(taskId);
            } else {
                mostrarMensagem(data.message, 'error');
            }
        })
        .catch(error => {
            console.error('Erro:', error);
            mostrarMensagem('Erro ao parar tarefa', 'error');
        });
    }
}

function concluirTarefa(taskId) {
    if (confirm('Deseja marcar esta tarefa como concluída?')) {
        fetch(`/task/${taskId}/concluir/`, {
            method: 'POST',
            headers: {
                'X-CSRFToken': getCookie('csrftoken'),
                'Content-Type': 'application/json',
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                mostrarMensagem(data.message, 'success');
                atualizarInterfaceTempo(taskId, data);
                pararCronometro(taskId);
                // Recarregar a página para mostrar o novo status
                setTimeout(() => {
                    window.location.reload();
                }, 1500);
            } else {
                mostrarMensagem(data.message, 'error');
            }
        })
        .catch(error => {
            console.error('Erro:', error);
            mostrarMensagem('Erro ao concluir tarefa', 'error');
        });
    }
}

// Funções auxiliares
function getCookie(name) {
    let cookieValue = null;
    if (document.cookie && document.cookie !== '') {
        const cookies = document.cookie.split(';');
        for (let i = 0; i < cookies.length; i++) {
            const cookie = cookies[i].trim();
            if (cookie.substring(0, name.length + 1) === (name + '=')) {
                cookieValue = decodeURIComponent(cookie.substring(name.length + 1));
                break;
            }
        }
    }
    return cookieValue;
}

function mostrarMensagem(mensagem, tipo) {
    // Criar toast de notificação
    const toastContainer = document.getElementById('toast-container') || criarToastContainer();
    
    const toast = document.createElement('div');
    toast.className = `toast align-items-center text-white bg-${tipo === 'success' ? 'success' : 'danger'} border-0`;
    toast.setAttribute('role', 'alert');
    toast.setAttribute('aria-live', 'assertive');
    toast.setAttribute('aria-atomic', 'true');
    
    toast.innerHTML = `
        <div class="d-flex">
            <div class="toast-body">
                ${mensagem}
            </div>
            <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
        </div>
    `;
    
    toastContainer.appendChild(toast);
    
    const bsToast = new bootstrap.Toast(toast);
    bsToast.show();
    
    // Remover toast após ser escondido
    toast.addEventListener('hidden.bs.toast', () => {
        toast.remove();
    });
}

function criarToastContainer() {
    const container = document.createElement('div');
    container.id = 'toast-container';
    container.className = 'toast-container position-fixed top-0 end-0 p-3';
    container.style.zIndex = '9999';
    document.body.appendChild(container);
    return container;
}

function atualizarInterfaceTempo(taskId, data) {
    // Atualizar tempos
    const tempoTotalElement = document.querySelector(`[data-task-id="${taskId}"] .tempo-total`);
    const tempoHojeElement = document.querySelector(`[data-task-id="${taskId}"] .tempo-hoje`);
    
    if (tempoTotalElement && data.tempo_total) {
        tempoTotalElement.textContent = data.tempo_total;
    }
    if (tempoHojeElement && data.tempo_hoje) {
        tempoHojeElement.textContent = data.tempo_hoje;
    }
    
    // Atualizar status
    const statusElement = document.querySelector(`[data-task-id="${taskId}"] .status-trabalho`);
    if (statusElement && data.status) {
        if (data.esta_trabalhando) {
            statusElement.innerHTML = '<i class="bi bi-play-circle"></i> Trabalhando';
            statusElement.className = 'text-success status-trabalho';
        } else if (data.status === 'Pausada') {
            statusElement.innerHTML = '<i class="bi bi-pause-circle"></i> Pausada';
            statusElement.className = 'text-warning status-trabalho';
        } else {
            statusElement.innerHTML = '<i class="bi bi-stop-circle"></i> Parada';
            statusElement.className = 'text-secondary status-trabalho';
        }
    }
    
    // Atualizar controles
    atualizarControlesTempo(taskId, data);
}

function atualizarControlesTempo(taskId, data) {
    const controlesElement = document.querySelector(`[data-task-id="${taskId}"] .controles-tempo`);
    
    if (!controlesElement) return;
    
    if (data.esta_trabalhando) {
        controlesElement.innerHTML = `
            <button class="btn btn-warning btn-sm" onclick="pausarTarefa(${taskId})" title="Pausar">
                <i class="bi bi-pause-circle"></i>
            </button>
            <button class="btn btn-danger btn-sm" onclick="pararTarefa(${taskId})" title="Parar">
                <i class="bi bi-stop-circle"></i>
            </button>
        `;
    } else if (data.status === 'Pausada') {
        controlesElement.innerHTML = `
            <button class="btn btn-info btn-sm" onclick="retomarTarefa(${taskId})" title="Retomar">
                <i class="bi bi-play-circle"></i>
            </button>
            <button class="btn btn-danger btn-sm" onclick="pararTarefa(${taskId})" title="Parar">
                <i class="bi bi-stop-circle"></i>
            </button>
        `;
    } else {
        controlesElement.innerHTML = `
            <button class="btn btn-success btn-sm" onclick="iniciarTarefa(${taskId})" title="Iniciar Trabalho">
                <i class="bi bi-play-circle"></i>
            </button>
        `;
    }
}

function iniciarCronometro(taskId) {
    const cronometroElement = document.getElementById(`cronometro-${taskId}`);
    
    if (cronometroElement && !cronometros[taskId]) {
        const tempoInicio = new Date();
        cronometros[taskId] = {
            element: cronometroElement,
            inicio: tempoInicio,
            interval: setInterval(() => {
                const agora = new Date();
                const diferenca = agora - tempoInicio;
                const horas = Math.floor(diferenca / 3600000);
                const minutos = Math.floor((diferenca % 3600000) / 60000);
                const segundos = Math.floor((diferenca % 60000) / 1000);
                
                cronometroElement.textContent = `${horas.toString().padStart(2, '0')}:${minutos.toString().padStart(2, '0')}:${segundos.toString().padStart(2, '0')}`;
            }, 1000)
        };
    }
}

function pararCronometro(taskId) {
    if (cronometros[taskId]) {
        clearInterval(cronometros[taskId].interval);
        delete cronometros[taskId];
        
        const cronometroElement = document.getElementById(`cronometro-${taskId}`);
        if (cronometroElement) {
            cronometroElement.textContent = '00:00:00';
        }
    }
}

// Função para obter informações de tempo em tempo real
function obterTempoTarefa(taskId) {
    fetch(`/task/${taskId}/tempo/`, {
        method: 'GET',
        headers: {
            'X-CSRFToken': getCookie('csrftoken'),
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            atualizarInterfaceTempo(taskId, data);
        }
    })
    .catch(error => {
        console.error('Erro ao obter tempo da tarefa:', error);
    });
}

// Inicialização
document.addEventListener('DOMContentLoaded', function() {
    // Inicializar cronômetros para tarefas que estão trabalhando
    const tarefasTrabalhando = document.querySelectorAll('[data-task-id][data-trabalhando="true"]');
    tarefasTrabalhando.forEach(element => {
        const taskId = element.getAttribute('data-task-id');
        iniciarCronometro(taskId);
    });
    
    // Atualizar tempos a cada 30 segundos
    setInterval(() => {
        const tarefas = document.querySelectorAll('[data-task-id]');
        tarefas.forEach(element => {
            const taskId = element.getAttribute('data-task-id');
            obterTempoTarefa(taskId);
        });
    }, 30000);
}); 