/**
 * Controle de Tempo Melhorado para Tarefas
 * Versão que funciona sem recarregar a página
 */

class TaskTimeController {
    constructor(taskId) {
        this.taskId = taskId;
        this.cronometroInterval = null;
        this.tempoInicioSessao = null;
        this.isWorking = false;
        
        this.init();
    }
    
    init() {
        console.log(`🔧 Inicializando controlador de tempo para tarefa ${this.taskId}`);
        this.criarToastContainer();
        // Verificar estado inicial e sincronizar
        this.updateFromServer().then(() => {
            console.log('🔄 Estado inicial sincronizado');
        });
    }
    
    // Funções principais de controle
    async iniciarTarefa() {
        if (confirm('Deseja iniciar o trabalho nesta tarefa?')) {
            try {
                const response = await fetch(`/tasks/${this.taskId}/iniciar/`, {
                    method: 'POST',
                    headers: {
                        'X-CSRFToken': this.getCookie('csrftoken'),
                        'Content-Type': 'application/json',
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    this.mostrarMensagem(data.message, 'success');
                    this.updateInterface(data);
                    this.iniciarCronometro();
                } else {
                    this.mostrarMensagem(data.message, 'error');
                }
            } catch (error) {
                console.error('Erro:', error);
                this.mostrarMensagem('Erro ao iniciar tarefa', 'error');
            }
        }
    }
    
    async pausarTarefa() {
        if (confirm('Deseja pausar o trabalho nesta tarefa?')) {
            try {
                const response = await fetch(`/tasks/${this.taskId}/pausar/`, {
                    method: 'POST',
                    headers: {
                        'X-CSRFToken': this.getCookie('csrftoken'),
                        'Content-Type': 'application/json',
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    this.mostrarMensagem(data.message, 'success');
                    this.updateInterface(data);
                    this.pararCronometro();
                    
                    // Não zerar a sessão, apenas parar o cronômetro
                    const tempoSessaoElement = document.getElementById('tempo-sessao');
                    if (tempoSessaoElement) {
                        tempoSessaoElement.textContent = '00:00:00';
                    }
                    
                    // Atualizar interface após pausar
                    setTimeout(() => {
                        this.updateFromServer();
                    }, 100);
                } else {
                    this.mostrarMensagem(data.message, 'error');
                    console.error('Erro ao pausar:', data.message);
                }
            } catch (error) {
                console.error('Erro:', error);
                this.mostrarMensagem('Erro ao pausar tarefa', 'error');
            }
        }
    }
    
    async retomarTarefa() {
        if (confirm('Deseja retomar o trabalho nesta tarefa?')) {
            try {
                const response = await fetch(`/tasks/${this.taskId}/retomar/`, {
                    method: 'POST',
                    headers: {
                        'X-CSRFToken': this.getCookie('csrftoken'),
                        'Content-Type': 'application/json',
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    this.mostrarMensagem(data.message, 'success');
                    this.updateInterface(data);
                    this.iniciarCronometro();
                } else {
                    this.mostrarMensagem(data.message, 'error');
                }
            } catch (error) {
                console.error('Erro:', error);
                this.mostrarMensagem('Erro ao retomar tarefa', 'error');
            }
        }
    }
    

    
    async concluirTarefa() {
        if (confirm('Deseja marcar esta tarefa como concluída?')) {
            try {
                const response = await fetch(`/tasks/${this.taskId}/concluir/`, {
                    method: 'POST',
                    headers: {
                        'X-CSRFToken': this.getCookie('csrftoken'),
                        'Content-Type': 'application/json',
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    this.mostrarMensagem(data.message, 'success');
                    this.updateInterface(data);
                    this.pararCronometro();
                    // Recarregar a página para mostrar o novo status
                    setTimeout(() => {
                        window.location.reload();
                    }, 1500);
                } else {
                    this.mostrarMensagem(data.message, 'error');
                }
            } catch (error) {
                console.error('Erro:', error);
                this.mostrarMensagem('Erro ao concluir tarefa', 'error');
            }
        }
    }
    
    async resetTarefa() {
        if (confirm('Deseja resetar o estado da tarefa? Isso irá parar qualquer trabalho em andamento.')) {
            try {
                const response = await fetch(`/tasks/${this.taskId}/reset/`, {
                    method: 'POST',
                    headers: {
                        'X-CSRFToken': this.getCookie('csrftoken'),
                        'Content-Type': 'application/json',
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    this.mostrarMensagem(data.message, 'success');
                    this.updateInterface(data);
                    this.pararCronometro();
                    // Recarregar a página para mostrar o novo status
                    setTimeout(() => {
                        window.location.reload();
                    }, 1500);
                } else {
                    this.mostrarMensagem(data.message, 'error');
                }
            } catch (error) {
                console.error('Erro:', error);
                this.mostrarMensagem('Erro ao resetar tarefa', 'error');
            }
        }
    }
    
    // Funções auxiliares
    getCookie(name) {
        let cookieValue = null;
        if (document.cookie && document.cookie !== '') {
            const cookies = document.cookie.split(';');
            for (let i = 0; i < cookies.length; i++) {
                const cookie = cookies[i].trim();
                if (cookie.substring(0, name.length + 1) === (name + '=')) {
                    cookieValue = decodeURIComponent(cookie.substring(name.length + 1));
                    break;
                }
            }
        }
        return cookieValue;
    }
    
    mostrarMensagem(mensagem, tipo) {
        // Criar toast de notificação
        const toastContainer = document.getElementById('toast-container') || this.criarToastContainer();
        
        const toast = document.createElement('div');
        toast.className = `toast align-items-center text-white bg-${tipo === 'success' ? 'success' : 'danger'} border-0`;
        toast.setAttribute('role', 'alert');
        toast.setAttribute('aria-live', 'assertive');
        toast.setAttribute('aria-atomic', 'true');
        
        toast.innerHTML = `
            <div class="d-flex">
                <div class="toast-body">
                    ${mensagem}
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
            </div>
        `;
        
        toastContainer.appendChild(toast);
        
        const bsToast = new bootstrap.Toast(toast);
        bsToast.show();
        
        // Remover toast após ser escondido
        toast.addEventListener('hidden.bs.toast', () => {
            toast.remove();
        });
    }
    
    criarToastContainer() {
        const container = document.createElement('div');
        container.id = 'toast-container';
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        container.style.zIndex = '9999';
        document.body.appendChild(container);
        return container;
    }
    
    updateInterface(data) {
        console.log('🔄 Atualizando interface com dados:', data);
        
        // Atualizar tempos
        if (data.tempo_total) {
            const tempoTotalElement = document.getElementById('tempo-total');
            if (tempoTotalElement) {
                tempoTotalElement.textContent = data.tempo_total;
            }
        }
        
        if (data.tempo_hoje) {
            const tempoHojeElement = document.getElementById('tempo-hoje');
            if (tempoHojeElement) {
                tempoHojeElement.textContent = data.tempo_hoje;
            }
        }
        

        
        // Atualizar cronômetro da sessão com tempo do servidor
        this.updateCronometro(data.esta_trabalhando, data.tempo_sessao);
        
        // Atualizar status
        this.updateStatus(data);
        
        // Atualizar controles
        this.updateControles(data);
        
        // Gerenciar cronômetro baseado no estado
        if (data.esta_trabalhando && !this.isWorking) {
            console.log('🔄 Iniciando cronômetro via updateInterface');
            this.iniciarCronometro();
        } else if (!data.esta_trabalhando && this.isWorking) {
            console.log('🔄 Parando cronômetro via updateInterface');
            this.pararCronometro();
        }
    }
    
    updateCronometro(estaTrabalhando, tempoSessaoServidor = null) {
        const tempoSessaoElement = document.getElementById('tempo-sessao');
        if (tempoSessaoElement) {
            if (estaTrabalhando) {
                // Criar ou atualizar o elemento do cronômetro
                let cronometroElement = document.getElementById('cronometro');
                if (!cronometroElement) {
                    cronometroElement = document.createElement('span');
                    cronometroElement.id = 'cronometro';
                    cronometroElement.className = 'text-success fw-bold';
                    tempoSessaoElement.innerHTML = '';
                    tempoSessaoElement.appendChild(cronometroElement);
                    console.log('🔄 Elemento do cronômetro criado');
                }
                // Mostrar o cronômetro
                cronometroElement.style.display = 'inline';
                
                // Se temos tempo do servidor, usar ele
                if (tempoSessaoServidor) {
                    cronometroElement.textContent = tempoSessaoServidor;
                    console.log('🔄 Cronômetro sincronizado com servidor:', tempoSessaoServidor);
                } else if (!this.isWorking) {
                    cronometroElement.textContent = '00:00:00';
                }
            } else {
                // Esconder o cronômetro se não estiver trabalhando (não remover)
                const cronometroElement = document.getElementById('cronometro');
                if (cronometroElement) {
                    cronometroElement.style.display = 'none';
                    console.log('🔄 Elemento do cronômetro escondido');
                }
                // Não zerar o tempo da sessão quando pausado, apenas esconder o cronômetro
                // O tempo será mostrado como 00:00:00 apenas quando parar completamente
            }
        }
    }
    
    updateStatus(data) {
        const statusElement = document.getElementById('status-trabalho');
        if (statusElement && data.status) {
            if (data.esta_trabalhando) {
                statusElement.innerHTML = '<i class="bi bi-play-circle"></i> Trabalhando';
                statusElement.className = 'text-success';
            } else if (data.status === 'Pausada') {
                statusElement.innerHTML = '<i class="bi bi-pause-circle"></i> Pausada';
                statusElement.className = 'text-warning';
            } else {
                statusElement.innerHTML = '<i class="bi bi-stop-circle"></i> Parada';
                statusElement.className = 'text-secondary';
            }
        }
    }
    
    updateControles(data) {
        const controles = document.getElementById('controles-tempo');
        if (!controles) return;
        
        if (data.esta_trabalhando) {
            controles.innerHTML = `
                <button class="btn btn-warning" onclick="taskController.pausarTarefa()" id="btn-pausar">
                    <i class="bi bi-pause-circle"></i> Pausar
                </button>
            `;
        } else if (data.status === 'Pausada') {
            controles.innerHTML = `
                <button class="btn btn-info" onclick="taskController.retomarTarefa()" id="btn-retomar">
                    <i class="bi bi-play-circle"></i> Retomar
                </button>
            `;
        } else {
            controles.innerHTML = `
                <button class="btn btn-success" onclick="taskController.iniciarTarefa()" id="btn-iniciar">
                    <i class="bi bi-play-circle"></i> Iniciar Trabalho
                </button>
            `;
        }
    }
    
    iniciarCronometro() {
        // Parar cronômetro anterior se existir
        this.pararCronometro();
        
        // Garantir que o elemento do cronômetro existe
        let cronometroElement = document.getElementById('cronometro');
        if (!cronometroElement) {
            const tempoSessaoElement = document.getElementById('tempo-sessao');
            if (tempoSessaoElement) {
                cronometroElement = document.createElement('span');
                cronometroElement.id = 'cronometro';
                cronometroElement.className = 'text-success fw-bold';
                tempoSessaoElement.innerHTML = '';
                tempoSessaoElement.appendChild(cronometroElement);
                console.log('🔄 Elemento do cronômetro criado em iniciarCronometro');
            }
        }
        
        if (cronometroElement) {
            console.log('⏱️ Iniciando cronômetro...');
            this.isWorking = true;
            
            // Obter tempo atual do servidor para sincronizar
            this.updateFromServer().then(() => {
                // Iniciar atualização periódica do cronômetro
                this.cronometroInterval = setInterval(() => {
                    // Atualizar com dados do servidor a cada segundo
                    this.updateFromServer();
                }, 1000);
            });
        } else {
            console.warn('⚠️ Elemento do cronômetro não encontrado e não foi possível criar');
        }
    }
    
    pararCronometro() {
        if (this.cronometroInterval) {
            clearInterval(this.cronometroInterval);
            this.cronometroInterval = null;
            this.isWorking = false;
            console.log('⏹️ Cronômetro parado');
        }
        
        // Não remover o elemento, apenas parar o intervalo
        // Não zerar o tempo automaticamente - isso será feito apenas quando parar completamente
        const cronometroElement = document.getElementById('cronometro');
        if (cronometroElement) {
            cronometroElement.style.display = 'none';
        }
    }
    
    async updateFromServer() {
        try {
            const response = await fetch(`/tasks/${this.taskId}/tempo/`, {
                method: 'GET',
                headers: {
                    'X-CSRFToken': this.getCookie('csrftoken'),
                }
            });
            
            const data = await response.json();
            
            if (data.success) {
                this.updateInterface(data);
                
                // Sincronizar cronômetro se estiver trabalhando
                if (data.esta_trabalhando && !this.isWorking) {
                    console.log('🔄 Sincronizando cronômetro com servidor...');
                    this.iniciarCronometro();
                } else if (!data.esta_trabalhando && this.isWorking) {
                    this.pararCronometro();
                } else if (data.esta_trabalhando && this.isWorking) {
                    // Se já está trabalhando, apenas atualizar a interface
                    console.log('🔄 Atualizando interface (cronômetro já ativo)');
                }
            }
        } catch (error) {
            console.error('Erro ao obter tempo da tarefa:', error);
        }
    }
    

}

// Variável global para o controlador
let taskController = null;

// Inicialização quando a página carrega
document.addEventListener('DOMContentLoaded', function() {
    // Extrair ID da tarefa da URL
    const url = window.location.pathname;
    const match = url.match(/\/tasks\/(\d+)\//);
    
    if (match) {
        const taskId = match[1];
        taskController = new TaskTimeController(taskId);
        console.log(`🚀 Controlador de tempo inicializado para tarefa ${taskId}`);
    }
}); 