from django.contrib import admin
from django.utils.html import format_html
from django.urls import reverse
from django.utils.safestring import mark_safe
from .models import SystemLog
from django.utils import timezone


@admin.register(SystemLog)
class SystemLogAdmin(admin.ModelAdmin):
    """
    Admin para o modelo SystemLog
    Acesso restrito apenas a superusuários
    """
    
    list_display = [
        'timestamp', 
        'user_display', 
        'action_display', 
        'severity_display', 
        'description_short', 
        'ip_address'
    ]
    
    list_filter = [
        'action',
        'severity',
        'timestamp',
        'user',
        'content_type',
    ]
    
    search_fields = [
        'description',
        'user__username',
        'user__first_name',
        'user__last_name',
        'ip_address',
        'url',
    ]
    
    readonly_fields = [
        'timestamp',
        'user',
        'action',
        'severity',
        'content_type',
        'object_id',
        'description',
        'details_display',
        'ip_address',
        'user_agent',
        'session_id',
        'url',
        'method',
    ]
    
    fieldsets = (
        ('Informações Básicas', {
            'fields': ('timestamp', 'user', 'action', 'severity')
        }),
        ('Objeto Afetado', {
            'fields': ('content_type', 'object_id'),
            'classes': ('collapse',)
        }),
        ('Detalhes da Ação', {
            'fields': ('description', 'details_display')
        }),
        ('Informações do Sistema', {
            'fields': ('ip_address', 'user_agent', 'session_id'),
            'classes': ('collapse',)
        }),
        ('Informações da Requisição', {
            'fields': ('url', 'method'),
            'classes': ('collapse',)
        }),
    )
    
    def user_display(self, obj):
        """Exibe o usuário com link para o admin"""
        if obj.user:
            url = reverse('admin:auth_user_change', args=[obj.user.id])
            return format_html('<a href="{}">{}</a>', url, obj.user.get_full_name() or obj.user.username)
        return "Usuário Anônimo"
    user_display.short_description = "Usuário"
    user_display.admin_order_field = 'user__username'
    
    def action_display(self, obj):
        """Exibe a ação com cor baseada na severidade"""
        colors = {
            'LOW': '#28a745',
            'MEDIUM': '#ffc107',
            'HIGH': '#fd7e14',
            'CRITICAL': '#dc3545'
        }
        color = colors.get(obj.severity, '#6c757d')
        return format_html(
            '<span style="color: {}; font-weight: bold;">{}</span>',
            color,
            obj.get_action_display()
        )
    action_display.short_description = "Ação"
    action_display.admin_order_field = 'action'
    
    def severity_display(self, obj):
        """Exibe a severidade com cor"""
        colors = {
            'LOW': '#28a745',
            'MEDIUM': '#ffc107',
            'HIGH': '#fd7e14',
            'CRITICAL': '#dc3545'
        }
        color = colors.get(obj.severity, '#6c757d')
        return format_html(
            '<span style="color: {}; font-weight: bold;">{}</span>',
            color,
            obj.get_severity_display()
        )
    severity_display.short_description = "Severidade"
    severity_display.admin_order_field = 'severity'
    
    def description_short(self, obj):
        """Exibe descrição truncada"""
        return obj.description[:100] + "..." if len(obj.description) > 100 else obj.description
    description_short.short_description = "Descrição"
    
    def details_display(self, obj):
        """Exibe os detalhes formatados"""
        if obj.details:
            return format_html('<pre>{}</pre>', obj.get_details_display())
        return "Nenhum detalhe adicional"
    details_display.short_description = "Detalhes Adicionais"
    
    def has_add_permission(self, request):
        """Impede a criação manual de logs"""
        return False
    
    def has_change_permission(self, request, obj=None):
        """Impede a edição de logs"""
        return False
    
    def has_delete_permission(self, request, obj=None):
        """Permite exclusão apenas para superusuários"""
        return request.user.is_superuser
    
    def get_queryset(self, request):
        """Filtra logs por superusuário"""
        qs = super().get_queryset(request)
        if not request.user.is_superuser:
            return qs.none()
        return qs
    
    def changelist_view(self, request, extra_context=None):
        """Adiciona contexto extra para estatísticas"""
        extra_context = extra_context or {}
        
        if request.user.is_superuser:
            qs = self.get_queryset(request)
            extra_context.update({
                'total_logs': qs.count(),
                'today_logs': qs.filter(timestamp__date=timezone.now().date()).count(),
                'critical_logs': qs.filter(severity='CRITICAL').count(),
                'error_logs': qs.filter(action='ERROR').count(),
            })
        
        return super().changelist_view(request, extra_context)
    
    class Media:
        css = {
            'all': ('admin/css/system_logs.css',)
        }
