from django.core.management.base import BaseCommand
from system_logs.models import SystemLog


class Command(BaseCommand):
    help = 'Limpa logs de erro específicos (como erros 404 do Chrome DevTools)'
    
    def add_arguments(self, parser):
        parser.add_argument(
            '--dry-run',
            action='store_true',
            help='Mostra quantos logs seriam removidos sem realmente removê-los'
        )
        
        parser.add_argument(
            '--force',
            action='store_true',
            help='Força a remoção sem confirmação'
        )
    
    def handle(self, *args, **options):
        dry_run = options['dry_run']
        force = options['force']
        
        # Padrões de erro que devem ser removidos
        error_patterns = [
            '/.well-known/appspecific/com.chrome.devtools.json',
            '/favicon.ico',
            '/robots.txt',
            '/sitemap.xml',
        ]
        
        # Contar logs que serão removidos
        logs_to_remove = SystemLog.objects.filter(
            action='ERROR',
            description__icontains='Erro HTTP 404'
        )
        
        # Filtrar por padrões específicos
        filtered_logs = logs_to_remove.filter(
            description__icontains='/.well-known/'
        )
        
        count = filtered_logs.count()
        
        if count == 0:
            self.stdout.write(
                self.style.SUCCESS('Nenhum log de erro encontrado para remoção.')
            )
            return
        
        self.stdout.write(f'Encontrados {count} logs de erro para remoção.')
        
        if dry_run:
            self.stdout.write(
                self.style.WARNING(f'DRY RUN: {count} logs seriam removidos')
            )
            return
        
        # Confirma a operação
        if not force:
            confirm = input(f'\nTem certeza que deseja remover {count} logs de erro? (y/N): ')
            if confirm.lower() != 'y':
                self.stdout.write('Operação cancelada.')
                return
        
        # Remove os logs
        try:
            filtered_logs.delete()
            self.stdout.write(
                self.style.SUCCESS(f'Sucesso! {count} logs de erro foram removidos.')
            )
        except Exception as e:
            self.stdout.write(
                self.style.ERROR(f'Erro ao remover logs: {str(e)}')
            ) 