from django.core.management.base import BaseCommand
from django.utils import timezone
from datetime import timedelta
from system_logs.models import SystemLog


class Command(BaseCommand):
    help = 'Limpa logs antigos do sistema'
    
    def add_arguments(self, parser):
        parser.add_argument(
            '--days',
            type=int,
            default=90,
            help='Número de dias para manter os logs (padrão: 90)'
        )
        
        parser.add_argument(
            '--severity',
            type=str,
            choices=['LOW', 'MEDIUM', 'HIGH', 'CRITICAL'],
            help='Limpar apenas logs de uma severidade específica'
        )
        
        parser.add_argument(
            '--action',
            type=str,
            help='Limpar apenas logs de uma ação específica'
        )
        
        parser.add_argument(
            '--dry-run',
            action='store_true',
            help='Mostra quantos logs seriam removidos sem realmente removê-los'
        )
        
        parser.add_argument(
            '--force',
            action='store_true',
            help='Força a remoção sem confirmação'
        )
    
    def handle(self, *args, **options):
        days = options['days']
        severity = options['severity']
        action = options['action']
        dry_run = options['dry_run']
        force = options['force']
        
        # Calcula a data limite
        cutoff_date = timezone.now() - timedelta(days=days)
        
        # Constrói a query
        queryset = SystemLog.objects.filter(timestamp__lt=cutoff_date)
        
        if severity:
            queryset = queryset.filter(severity=severity)
        
        if action:
            queryset = queryset.filter(action=action)
        
        # Conta quantos logs serão removidos
        count = queryset.count()
        
        if count == 0:
            self.stdout.write(
                self.style.SUCCESS('Nenhum log antigo encontrado para remoção.')
            )
            return
        
        # Mostra informações sobre a operação
        self.stdout.write(f'Encontrados {count} logs antigos (mais de {days} dias)')
        
        if severity:
            self.stdout.write(f'Filtro de severidade: {severity}')
        
        if action:
            self.stdout.write(f'Filtro de ação: {action}')
        
        if dry_run:
            self.stdout.write(
                self.style.WARNING(f'DRY RUN: {count} logs seriam removidos')
            )
            return
        
        # Confirma a operação
        if not force:
            confirm = input(f'\nTem certeza que deseja remover {count} logs? (y/N): ')
            if confirm.lower() != 'y':
                self.stdout.write('Operação cancelada.')
                return
        
        # Remove os logs
        try:
            queryset.delete()
            self.stdout.write(
                self.style.SUCCESS(f'Sucesso! {count} logs foram removidos.')
            )
        except Exception as e:
            self.stdout.write(
                self.style.ERROR(f'Erro ao remover logs: {str(e)}')
            )
    
    def show_statistics(self):
        """Mostra estatísticas dos logs"""
        total_logs = SystemLog.objects.count()
        
        # Logs por severidade
        severity_stats = {}
        for severity_code, severity_name in SystemLog.SEVERITY_CHOICES:
            count = SystemLog.objects.filter(severity=severity_code).count()
            severity_stats[severity_name] = count
        
        # Logs por ação
        action_stats = {}
        for action_code, action_name in SystemLog.ACTION_CHOICES:
            count = SystemLog.objects.filter(action=action_code).count()
            action_stats[action_name] = count
        
        # Logs antigos (mais de 30 dias)
        old_logs = SystemLog.objects.filter(
            timestamp__lt=timezone.now() - timedelta(days=30)
        ).count()
        
        self.stdout.write('\n=== Estatísticas dos Logs ===')
        self.stdout.write(f'Total de logs: {total_logs}')
        self.stdout.write(f'Logs antigos (>30 dias): {old_logs}')
        
        self.stdout.write('\nPor Severidade:')
        for severity_name, count in severity_stats.items():
            self.stdout.write(f'  {severity_name}: {count}')
        
        self.stdout.write('\nPor Ação:')
        for action_name, count in action_stats.items():
            if count > 0:
                self.stdout.write(f'  {action_name}: {count}') 