from django.core.management.base import BaseCommand
from django.utils import timezone
from datetime import datetime, timedelta
from system_logs.models import SystemLog
import json
import csv
import os


class Command(BaseCommand):
    help = 'Exporta logs do sistema para arquivo CSV ou JSON'
    
    def add_arguments(self, parser):
        parser.add_argument(
            '--format',
            type=str,
            choices=['csv', 'json'],
            default='csv',
            help='Formato de exportação (csv ou json)'
        )
        
        parser.add_argument(
            '--output',
            type=str,
            help='Nome do arquivo de saída'
        )
        
        parser.add_argument(
            '--days',
            type=int,
            help='Exportar apenas logs dos últimos N dias'
        )
        
        parser.add_argument(
            '--severity',
            type=str,
            choices=['LOW', 'MEDIUM', 'HIGH', 'CRITICAL'],
            help='Exportar apenas logs de uma severidade específica'
        )
        
        parser.add_argument(
            '--action',
            type=str,
            help='Exportar apenas logs de uma ação específica'
        )
        
        parser.add_argument(
            '--user',
            type=str,
            help='Exportar apenas logs de um usuário específico'
        )
    
    def handle(self, *args, **options):
        format_type = options['format']
        output_file = options['output']
        days = options['days']
        severity = options['severity']
        action = options['action']
        user = options['user']
        
        # Constrói a query
        queryset = SystemLog.objects.all()
        
        if days:
            start_date = timezone.now() - timedelta(days=days)
            queryset = queryset.filter(timestamp__gte=start_date)
        
        if severity:
            queryset = queryset.filter(severity=severity)
        
        if action:
            queryset = queryset.filter(action=action)
        
        if user:
            queryset = queryset.filter(user__username=user)
        
        # Ordena por timestamp
        queryset = queryset.order_by('-timestamp')
        
        # Conta quantos logs serão exportados
        count = queryset.count()
        
        if count == 0:
            self.stdout.write(
                self.style.WARNING('Nenhum log encontrado para exportação.')
            )
            return
        
        # Gera nome do arquivo se não fornecido
        if not output_file:
            timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            output_file = f'system_logs_{timestamp}.{format_type}'
        
        # Exporta os logs
        if format_type == 'csv':
            self.export_csv(queryset, output_file)
        elif format_type == 'json':
            self.export_json(queryset, output_file)
        
        self.stdout.write(
            self.style.SUCCESS(f'Exportação concluída! {count} logs exportados para {output_file}')
        )
    
    def export_csv(self, queryset, output_file):
        """Exporta logs para CSV"""
        with open(output_file, 'w', newline='', encoding='utf-8') as csvfile:
            fieldnames = [
                'ID', 'Data/Hora', 'Usuário', 'Ação', 'Severidade', 
                'Descrição', 'IP', 'URL', 'Método', 'User Agent'
            ]
            
            writer = csv.DictWriter(csvfile, fieldnames=fieldnames)
            writer.writeheader()
            
            for log in queryset:
                writer.writerow({
                    'ID': log.id,
                    'Data/Hora': log.timestamp.strftime('%d/%m/%Y %H:%M:%S'),
                    'Usuário': log.user.get_full_name() if log.user else 'Anônimo',
                    'Ação': log.get_action_display(),
                    'Severidade': log.get_severity_display(),
                    'Descrição': log.description,
                    'IP': log.ip_address or '',
                    'URL': log.url or '',
                    'Método': log.method or '',
                    'User Agent': log.user_agent or '',
                })
    
    def export_json(self, queryset, output_file):
        """Exporta logs para JSON"""
        logs_data = []
        
        for log in queryset:
            log_data = {
                'id': log.id,
                'timestamp': log.timestamp.isoformat(),
                'user': {
                    'id': log.user.id if log.user else None,
                    'username': log.user.username if log.user else None,
                    'full_name': log.user.get_full_name() if log.user else None,
                } if log.user else None,
                'action': {
                    'code': log.action,
                    'display': log.get_action_display(),
                },
                'severity': {
                    'code': log.severity,
                    'display': log.get_severity_display(),
                },
                'description': log.description,
                'content_object': {
                    'type': log.content_type.model if log.content_type else None,
                    'id': log.object_id,
                } if log.content_type and log.object_id else None,
                'details': log.details,
                'ip_address': log.ip_address,
                'user_agent': log.user_agent,
                'session_id': log.session_id,
                'url': log.url,
                'method': log.method,
            }
            
            logs_data.append(log_data)
        
        with open(output_file, 'w', encoding='utf-8') as jsonfile:
            json.dump(logs_data, jsonfile, indent=2, ensure_ascii=False) 