from django.utils.deprecation import MiddlewareMixin
from django.contrib.contenttypes.models import ContentType
from .models import SystemLog


class SystemLogMiddleware(MiddlewareMixin):
    """
    Middleware para capturar automaticamente informações da requisição
    e facilitar o registro de logs do sistema
    """
    
    def process_request(self, request):
        """Captura informações da requisição"""
        # Adiciona informações da requisição ao request para uso posterior
        request.system_log_info = {
            'ip_address': self.get_client_ip(request),
            'user_agent': request.META.get('HTTP_USER_AGENT', ''),
            'session_id': request.session.session_key,
            'url': request.build_absolute_uri(),
            'method': request.method,
        }
    
    def get_client_ip(self, request):
        """Obtém o IP real do cliente"""
        x_forwarded_for = request.META.get('HTTP_X_FORWARDED_FOR')
        if x_forwarded_for:
            ip = x_forwarded_for.split(',')[0]
        else:
            ip = request.META.get('REMOTE_ADDR')
        return ip
    
    def process_response(self, request, response):
        """Processa a resposta e registra logs se necessário"""
        # Filtra requisições que não devem ser logadas
        if self.should_skip_logging(request):
            return response
        
        # Registra apenas erros 5xx (erros do servidor)
        if response.status_code >= 500:
            if hasattr(request, 'user') and request.user.is_authenticated:
                SystemLog.log_action(
                    user=request.user,
                    action='ERROR',
                    severity='CRITICAL',
                    description=f"Erro HTTP {response.status_code} em {request.path}",
                    ip_address=getattr(request, 'system_log_info', {}).get('ip_address'),
                    user_agent=getattr(request, 'system_log_info', {}).get('user_agent'),
                    session_id=getattr(request, 'system_log_info', {}).get('session_id'),
                    url=getattr(request, 'system_log_info', {}).get('url'),
                    method=getattr(request, 'system_log_info', {}).get('method'),
                )
        
        return response
    
    def should_skip_logging(self, request):
        """Verifica se a requisição deve ser ignorada no logging"""
        # URLs que devem ser ignoradas
        skip_patterns = [
            '/.well-known/',
            '/static/',
            '/media/',
            '/favicon.ico',
            '/robots.txt',
            '/sitemap.xml',
            '/__debug__/',
            '/admin/jsi18n/',
        ]
        
        path = request.path.lower()
        for pattern in skip_patterns:
            if pattern in path:
                return True
        
        # Ignora requisições de arquivos estáticos
        if path.endswith(('.css', '.js', '.png', '.jpg', '.jpeg', '.gif', '.ico', '.svg')):
            return True
        
        return False
    
    def process_exception(self, request, exception):
        """Processa exceções e registra logs de erro"""
        if hasattr(request, 'user') and request.user.is_authenticated:
            SystemLog.log_action(
                user=request.user,
                action='ERROR',
                severity='CRITICAL',
                description=f"Exceção: {str(exception)} em {request.path}",
                details={
                    'exception_type': type(exception).__name__,
                    'exception_message': str(exception),
                    'path': request.path,
                    'method': request.method,
                },
                ip_address=getattr(request, 'system_log_info', {}).get('ip_address'),
                user_agent=getattr(request, 'system_log_info', {}).get('user_agent'),
                session_id=getattr(request, 'system_log_info', {}).get('session_id'),
                url=getattr(request, 'system_log_info', {}).get('url'),
                method=getattr(request, 'system_log_info', {}).get('method'),
            ) 