from django.db import models
from django.contrib.auth.models import User
from django.contrib.contenttypes.models import ContentType
from django.contrib.contenttypes.fields import GenericForeignKey
from django.utils import timezone
import json


class SystemLog(models.Model):
    """
    Modelo para registrar todas as ações do sistema
    """
    
    ACTION_CHOICES = [
        ('CREATE', 'Criar'),
        ('UPDATE', 'Atualizar'),
        ('DELETE', 'Excluir'),
        ('LOGIN', 'Login'),
        ('LOGOUT', 'Logout'),
        ('VIEW', 'Visualizar'),
        ('DOWNLOAD', 'Download'),
        ('UPLOAD', 'Upload'),
        ('EXPORT', 'Exportar'),
        ('IMPORT', 'Importar'),
        ('PERMISSION_CHANGE', 'Alteração de Permissão'),
        ('SYSTEM_CONFIG', 'Configuração do Sistema'),
        ('ERROR', 'Erro'),
        ('WARNING', 'Aviso'),
        ('INFO', 'Informação'),
    ]
    
    SEVERITY_CHOICES = [
        ('LOW', 'Baixo'),
        ('MEDIUM', 'Médio'),
        ('HIGH', 'Alto'),
        ('CRITICAL', 'Crítico'),
    ]
    
    # Informações básicas
    timestamp = models.DateTimeField(auto_now_add=True, verbose_name="Data/Hora")
    user = models.ForeignKey(
        User, 
        on_delete=models.SET_NULL, 
        null=True, 
        blank=True, 
        verbose_name="Usuário"
    )
    action = models.CharField(
        max_length=50, 
        choices=ACTION_CHOICES, 
        verbose_name="Ação"
    )
    severity = models.CharField(
        max_length=20, 
        choices=SEVERITY_CHOICES, 
        default='MEDIUM', 
        verbose_name="Severidade"
    )
    
    # Informações do objeto afetado (opcional)
    content_type = models.ForeignKey(
        ContentType, 
        on_delete=models.CASCADE, 
        null=True, 
        blank=True, 
        verbose_name="Tipo de Conteúdo"
    )
    object_id = models.PositiveIntegerField(
        null=True, 
        blank=True, 
        verbose_name="ID do Objeto"
    )
    content_object = GenericForeignKey('content_type', 'object_id')
    
    # Detalhes da ação
    description = models.TextField(verbose_name="Descrição")
    details = models.JSONField(
        null=True, 
        blank=True, 
        verbose_name="Detalhes Adicionais"
    )
    
    # Informações do sistema
    ip_address = models.GenericIPAddressField(
        null=True, 
        blank=True, 
        verbose_name="Endereço IP"
    )
    user_agent = models.TextField(
        null=True, 
        blank=True, 
        verbose_name="User Agent"
    )
    session_id = models.CharField(
        max_length=100, 
        null=True, 
        blank=True, 
        verbose_name="ID da Sessão"
    )
    
    # Informações da URL
    url = models.URLField(
        max_length=500, 
        null=True, 
        blank=True, 
        verbose_name="URL"
    )
    method = models.CharField(
        max_length=10, 
        null=True, 
        blank=True, 
        verbose_name="Método HTTP"
    )
    
    class Meta:
        verbose_name = "Log do Sistema"
        verbose_name_plural = "Logs do Sistema"
        ordering = ['-timestamp']
        indexes = [
            models.Index(fields=['timestamp']),
            models.Index(fields=['user']),
            models.Index(fields=['action']),
            models.Index(fields=['severity']),
            models.Index(fields=['content_type', 'object_id']),
        ]
    
    def __str__(self):
        return f"{self.timestamp.strftime('%d/%m/%Y %H:%M:%S')} - {self.user} - {self.get_action_display()}"
    
    def get_details_display(self):
        """Retorna os detalhes formatados para exibição"""
        if self.details:
            return json.dumps(self.details, indent=2, ensure_ascii=False)
        return "Nenhum detalhe adicional"
    
    @classmethod
    def log_action(cls, user, action, description, **kwargs):
        """
        Método de classe para facilitar o registro de logs
        """
        return cls.objects.create(
            user=user,
            action=action,
            description=description,
            **kwargs
        )
