from django.contrib.contenttypes.models import ContentType
from django.contrib.auth.models import AnonymousUser
from .models import SystemLog


def log_system_action(request, action, description, severity='MEDIUM', obj=None, details=None):
    """
    Utilitário para registrar ações do sistema
    
    Args:
        request: Objeto request do Django
        action: Tipo de ação (CREATE, UPDATE, DELETE, etc.)
        description: Descrição da ação
        severity: Severidade (LOW, MEDIUM, HIGH, CRITICAL)
        obj: Objeto relacionado à ação (opcional)
        details: Detalhes adicionais em formato dict (opcional)
    """
    try:
        # Verifica se o usuário está autenticado
        user = request.user if hasattr(request, 'user') and not isinstance(request.user, AnonymousUser) else None
        
        # Obtém informações da requisição
        system_info = getattr(request, 'system_log_info', {})
        
        # Prepara dados do objeto relacionado
        content_type = None
        object_id = None
        
        if obj:
            content_type = ContentType.objects.get_for_model(obj)
            object_id = obj.id
        
        # Registra o log
        log = SystemLog.log_action(
            user=user,
            action=action,
            severity=severity,
            description=description,
            content_type=content_type,
            object_id=object_id,
            details=details,
            ip_address=system_info.get('ip_address'),
            user_agent=system_info.get('user_agent'),
            session_id=system_info.get('session_id'),
            url=system_info.get('url'),
            method=system_info.get('method'),
        )
        
        return log
    
    except Exception as e:
        # Em caso de erro, tenta registrar o erro sem usar o request
        try:
            SystemLog.log_action(
                user=None,
                action='ERROR',
                severity='CRITICAL',
                description=f"Erro ao registrar log: {str(e)}",
                details={
                    'original_action': action,
                    'original_description': description,
                    'error': str(e),
                }
            )
        except:
            # Se não conseguir registrar o erro, apenas ignora
            pass
        
        return None


def log_user_login(request, user, success=True):
    """Registra login/logout de usuário"""
    action = 'LOGIN' if success else 'ERROR'
    severity = 'MEDIUM' if success else 'HIGH'
    description = f"{'Login bem-sucedido' if success else 'Tentativa de login falhou'} para o usuário {user.username}"
    
    return log_system_action(request, action, description, severity)


def log_user_logout(request, user):
    """Registra logout de usuário"""
    description = f"Logout do usuário {user.username}"
    return log_system_action(request, 'LOGOUT', description, 'LOW')


def log_object_created(request, obj, description=None):
    """Registra criação de objeto"""
    if not description:
        description = f"Criado {obj._meta.verbose_name}: {str(obj)}"
    
    return log_system_action(request, 'CREATE', description, 'MEDIUM', obj)


def log_object_updated(request, obj, description=None, changes=None):
    """Registra atualização de objeto"""
    if not description:
        description = f"Atualizado {obj._meta.verbose_name}: {str(obj)}"
    
    details = {'changes': changes} if changes else None
    return log_system_action(request, 'UPDATE', description, 'MEDIUM', obj, details)


def log_object_deleted(request, obj, description=None):
    """Registra exclusão de objeto"""
    if not description:
        description = f"Excluído {obj._meta.verbose_name}: {str(obj)}"
    
    return log_system_action(request, 'DELETE', description, 'HIGH', obj)


def log_permission_change(request, user, permission, granted=True):
    """Registra alteração de permissão"""
    action = 'PERMISSION_CHANGE'
    description = f"{'Concedida' if granted else 'Revogada'} permissão '{permission}' para o usuário {user.username}"
    severity = 'HIGH'
    
    return log_system_action(request, action, description, severity)


def log_system_config(request, config_name, description):
    """Registra alteração de configuração do sistema"""
    return log_system_action(request, 'SYSTEM_CONFIG', description, 'HIGH')


def log_error(request, error_message, details=None):
    """Registra erro do sistema"""
    return log_system_action(request, 'ERROR', error_message, 'CRITICAL', details=details)


def log_warning(request, warning_message, details=None):
    """Registra aviso do sistema"""
    return log_system_action(request, 'WARNING', warning_message, 'MEDIUM', details=details)


def log_info(request, info_message, details=None):
    """Registra informação do sistema"""
    return log_system_action(request, 'INFO', info_message, 'LOW', details=details)


# Decorator para registrar automaticamente ações de views
def log_view_action(action, severity='MEDIUM'):
    """
    Decorator para registrar automaticamente ações de views
    
    Exemplo de uso:
    @log_view_action('VIEW', 'LOW')
    def my_view(request):
        # código da view
        pass
    """
    def decorator(view_func):
        def wrapper(request, *args, **kwargs):
            # Executa a view
            response = view_func(request, *args, **kwargs)
            
            # Registra o log
            description = f"Acesso à view {view_func.__name__}"
            log_system_action(request, action, description, severity)
            
            return response
        return wrapper
    return decorator 