from django.shortcuts import render, get_object_or_404
from django.contrib.auth.decorators import login_required, user_passes_test
from django.contrib.auth.mixins import LoginRequiredMixin, UserPassesTestMixin
from django.views.generic import ListView, DetailView
from django.core.paginator import Paginator
from django.db.models import Q
from django.utils import timezone
from datetime import datetime, timedelta
from .models import SystemLog


def is_superuser(user):
    """Verifica se o usuário é superusuário"""
    return user.is_superuser


@login_required
@user_passes_test(is_superuser)
def log_list(request):
    """
    View para listar logs do sistema
    Acesso restrito apenas a superusuários
    """
    # Parâmetros de filtro
    action = request.GET.get('action')
    severity = request.GET.get('severity')
    user_id = request.GET.get('user')
    date_from = request.GET.get('date_from')
    date_to = request.GET.get('date_to')
    search = request.GET.get('search')
    
    # Query base
    logs = SystemLog.objects.all()
    
    # Aplicar filtros
    if action:
        logs = logs.filter(action=action)
    
    if severity:
        logs = logs.filter(severity=severity)
    
    if user_id:
        logs = logs.filter(user_id=user_id)
    
    if date_from:
        try:
            date_from = datetime.strptime(date_from, '%Y-%m-%d').date()
            logs = logs.filter(timestamp__date__gte=date_from)
        except ValueError:
            pass
    
    if date_to:
        try:
            date_to = datetime.strptime(date_to, '%Y-%m-%d').date()
            logs = logs.filter(timestamp__date__lte=date_to)
        except ValueError:
            pass
    
    if search:
        logs = logs.filter(
            Q(description__icontains=search) |
            Q(user__username__icontains=search) |
            Q(user__first_name__icontains=search) |
            Q(user__last_name__icontains=search) |
            Q(ip_address__icontains=search)
        )
    
    # Paginação
    paginator = Paginator(logs, 50)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)
    
    # Estatísticas
    total_logs = logs.count()
    today_logs = logs.filter(timestamp__date=timezone.now().date()).count()
    critical_logs = logs.filter(severity='CRITICAL').count()
    error_logs = logs.filter(action='ERROR').count()
    
    context = {
        'page_obj': page_obj,
        'total_logs': total_logs,
        'today_logs': today_logs,
        'critical_logs': critical_logs,
        'error_logs': error_logs,
        'action_choices': SystemLog.ACTION_CHOICES,
        'severity_choices': SystemLog.SEVERITY_CHOICES,
        'current_filters': {
            'action': action,
            'severity': severity,
            'user_id': user_id,
            'date_from': date_from,
            'date_to': date_to,
            'search': search,
        }
    }
    
    return render(request, 'system_logs/list.html', context)


@login_required
@user_passes_test(is_superuser)
def log_detail(request, log_id):
    """
    View para detalhes de um log específico
    Acesso restrito apenas a superusuários
    """
    log = get_object_or_404(SystemLog, id=log_id)
    
    context = {
        'log': log,
    }
    
    return render(request, 'system_logs/detail.html', context)


@login_required
@user_passes_test(is_superuser)
def log_dashboard(request):
    """
    Dashboard com estatísticas dos logs
    Acesso restrito apenas a superusuários
    """
    # Estatísticas gerais
    total_logs = SystemLog.objects.count()
    today_logs = SystemLog.objects.filter(timestamp__date=timezone.now().date()).count()
    week_logs = SystemLog.objects.filter(
        timestamp__date__gte=timezone.now().date() - timedelta(days=7)
    ).count()
    month_logs = SystemLog.objects.filter(
        timestamp__date__gte=timezone.now().date() - timedelta(days=30)
    ).count()
    
    # Logs por severidade
    severity_stats = {}
    for severity_code, severity_name in SystemLog.SEVERITY_CHOICES:
        count = SystemLog.objects.filter(severity=severity_code).count()
        severity_stats[severity_name] = count
    
    # Logs por ação
    action_stats = {}
    for action_code, action_name in SystemLog.ACTION_CHOICES:
        count = SystemLog.objects.filter(action=action_code).count()
        action_stats[action_name] = count
    
    # Logs recentes
    recent_logs = SystemLog.objects.order_by('-timestamp')[:10]
    
    # Logs críticos
    critical_logs = SystemLog.objects.filter(severity='CRITICAL').order_by('-timestamp')[:5]
    
    context = {
        'total_logs': total_logs,
        'today_logs': today_logs,
        'week_logs': week_logs,
        'month_logs': month_logs,
        'severity_stats': severity_stats,
        'action_stats': action_stats,
        'recent_logs': recent_logs,
        'critical_logs': critical_logs,
    }
    
    return render(request, 'system_logs/dashboard.html', context)


class SystemLogListView(LoginRequiredMixin, UserPassesTestMixin, ListView):
    """
    View baseada em classe para listar logs
    """
    model = SystemLog
    template_name = 'system_logs/list.html'
    context_object_name = 'logs'
    paginate_by = 50
    ordering = ['-timestamp']
    
    def test_func(self):
        """Verifica se o usuário é superusuário"""
        return self.request.user.is_superuser
    
    def get_queryset(self):
        """Aplica filtros na queryset"""
        queryset = super().get_queryset()
        
        # Filtros
        action = self.request.GET.get('action')
        severity = self.request.GET.get('severity')
        search = self.request.GET.get('search')
        
        if action:
            queryset = queryset.filter(action=action)
        
        if severity:
            queryset = queryset.filter(severity=severity)
        
        if search:
            queryset = queryset.filter(
                Q(description__icontains=search) |
                Q(user__username__icontains=search)
            )
        
        return queryset
    
    def get_context_data(self, **kwargs):
        """Adiciona contexto extra"""
        context = super().get_context_data(**kwargs)
        context.update({
            'action_choices': SystemLog.ACTION_CHOICES,
            'severity_choices': SystemLog.SEVERITY_CHOICES,
        })
        return context


class SystemLogDetailView(LoginRequiredMixin, UserPassesTestMixin, DetailView):
    """
    View baseada em classe para detalhes de log
    """
    model = SystemLog
    template_name = 'system_logs/detail.html'
    context_object_name = 'log'
    
    def test_func(self):
        """Verifica se o usuário é superusuário"""
        return self.request.user.is_superuser
