from django.contrib import admin
from .models import Group, Task


@admin.register(Group)
class GroupAdmin(admin.ModelAdmin):
    """
    Admin para modelo Group.
    """
    list_display = ['nome', 'board', 'ordem', 'ativo']
    list_filter = ['ativo', 'board', 'created_at']
    search_fields = ['nome', 'descricao', 'board__nome']
    list_editable = ['ativo', 'ordem']
    ordering = ['board__nome', 'ordem', 'nome']
    
    fieldsets = (
        ('Informações Básicas', {
            'fields': ('nome', 'descricao', 'board', 'ativo')
        }),
        ('Organização', {
            'fields': ('ordem',)
        }),
        ('Metadados', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    
    readonly_fields = ['created_at', 'updated_at']


@admin.register(Task)
class TaskAdmin(admin.ModelAdmin):
    """
    Admin para modelo Task.
    """
    list_display = ['nome', 'group', 'responsavel', 'status', 'prioridade', 'cronograma_fim', 'progresso_percentual']
    list_filter = ['status', 'prioridade', 'group__board', 'responsavel', 'created_at']
    search_fields = ['nome', 'descricao', 'group__nome', 'responsavel__nome']
    list_editable = ['status', 'prioridade']
    ordering = ['group__board__nome', 'group__nome', 'cronograma_inicio']
    date_hierarchy = 'cronograma_inicio'
    
    fieldsets = (
        ('Informações Básicas', {
            'fields': ('nome', 'descricao', 'group', 'responsavel')
        }),
        ('Cronograma', {
            'fields': ('cronograma_inicio', 'cronograma_fim', 'status', 'prioridade')
        }),
        ('Esforço', {
            'fields': ('esforco_previsto', 'esforco_utilizado')
        }),
        ('Controle de Tempo', {
            'fields': ('tempo_iniciado', 'tempo_pausado', 'tempo_total_trabalhado'),
            'classes': ('collapse',)
        }),
        ('Metadados', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    
    readonly_fields = ['created_at', 'updated_at', 'tempo_total_trabalhado']
    filter_horizontal = ['dependencias']
    
    def progresso_percentual(self, obj):
        return f"{obj.progresso_percentual}%"
    progresso_percentual.short_description = 'Progresso'
    
    def get_queryset(self, request):
        return super().get_queryset(request).select_related('group', 'responsavel')

