from rest_framework import viewsets, status
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter

from .models import Group, Task
from .serializers import (
    GroupSerializer, 
    GroupDetailSerializer, 
    GroupListSerializer,
    TaskSerializer, 
    TaskDetailSerializer, 
    TaskListSerializer,
    TaskCreateSerializer,
    TaskUpdateStatusSerializer,
    TaskTimeControlSerializer
)


class GroupViewSet(viewsets.ModelViewSet):
    """
    ViewSet para operações CRUD de grupos.
    """
    
    queryset = Group.objects.all()
    permission_classes = [IsAuthenticated]
    filter_backends = [DjangoFilterBackend, SearchFilter, OrderingFilter]
    filterset_fields = ['ativo', 'board']
    search_fields = ['nome', 'descricao', 'board__nome']
    ordering_fields = ['nome', 'board__nome', 'ordem', 'created_at']
    ordering = ['board__nome', 'ordem', 'nome']

    def get_serializer_class(self):
        if self.action == 'retrieve':
            return GroupDetailSerializer
        if self.action == 'list':
            return GroupListSerializer
        return GroupSerializer

    def get_queryset(self):
        queryset = Group.objects.select_related('board').all()
        
        # Filtro por status ativo
        ativo = self.request.query_params.get('ativo', None)
        if ativo is not None:
            queryset = queryset.filter(ativo=ativo.lower() == 'true')
            
        # Filtro por board
        board_id = self.request.query_params.get('board_id', None)
        if board_id:
            queryset = queryset.filter(board_id=board_id)
            
        return queryset

    @action(detail=True, methods=['post'])
    def toggle_status(self, request, pk=None):
        """
        Alterna o status ativo/inativo do grupo.
        """
        group = self.get_object()
        group.ativo = not group.ativo
        group.save()
        
        serializer = self.get_serializer(group)
        return Response(serializer.data)

    @action(detail=True, methods=['get'])
    def tasks(self, request, pk=None):
        """
        Retorna as tarefas do grupo.
        """
        group = self.get_object()
        tasks = group.tasks.all().order_by('cronograma_inicio', 'nome')
        
        serializer = TaskListSerializer(tasks, many=True)
        return Response(serializer.data)

    @action(detail=True, methods=['patch'])
    def reorder(self, request, pk=None):
        """
        Reordena a posição do grupo.
        """
        group = self.get_object()
        nova_ordem = request.data.get('ordem')
        
        if nova_ordem is not None:
            try:
                nova_ordem = int(nova_ordem)
                group.ordem = nova_ordem
                group.save()
                
                serializer = self.get_serializer(group)
                return Response(serializer.data)
            except ValueError:
                return Response(
                    {"error": "Ordem deve ser um número inteiro"}, 
                    status=status.HTTP_400_BAD_REQUEST
                )
        
        return Response(
            {"error": "Campo 'ordem' é obrigatório"}, 
            status=status.HTTP_400_BAD_REQUEST
        )


class TaskViewSet(viewsets.ModelViewSet):
    """
    ViewSet para operações CRUD de tarefas.
    """
    
    queryset = Task.objects.all()
    permission_classes = [IsAuthenticated]
    filter_backends = [DjangoFilterBackend, SearchFilter, OrderingFilter]
    filterset_fields = ['status', 'prioridade', 'group', 'responsavel']
    search_fields = ['nome', 'descricao', 'group__nome', 'responsavel__nome']
    ordering_fields = ['nome', 'cronograma_inicio', 'cronograma_fim', 'prioridade', 'created_at']
    ordering = ['cronograma_inicio', 'nome']

    def get_serializer_class(self):
        if self.action == 'create':
            return TaskCreateSerializer
        if self.action == 'retrieve':
            return TaskDetailSerializer
        if self.action == 'list':
            return TaskListSerializer
        if self.action == 'update_status':
            return TaskUpdateStatusSerializer
        if self.action == 'time_control':
            return TaskTimeControlSerializer
        return TaskSerializer

    def get_queryset(self):
        queryset = Task.objects.select_related('group', 'responsavel').all()
        
        # Filtro por grupo
        group_id = self.request.query_params.get('group_id', None)
        if group_id:
            queryset = queryset.filter(group_id=group_id)
            
        # Filtro por board
        board_id = self.request.query_params.get('board_id', None)
        if board_id:
            queryset = queryset.filter(group__board_id=board_id)
            
        # Filtro por projeto
        project_id = self.request.query_params.get('project_id', None)
        if project_id:
            queryset = queryset.filter(group__board__project_id=project_id)
            
        # Filtro por responsável
        responsavel_id = self.request.query_params.get('responsavel_id', None)
        if responsavel_id:
            queryset = queryset.filter(responsavel_id=responsavel_id)
            
        # Filtro por tarefas atrasadas
        atrasadas = self.request.query_params.get('atrasadas', None)
        if atrasadas and atrasadas.lower() == 'true':
            from django.utils import timezone
            queryset = queryset.filter(
                cronograma_fim__lt=timezone.now().date(),
                status__in=['nao_iniciada', 'em_andamento', 'pausada']
            )
            
        # Filtro por tarefas que podem iniciar
        pode_iniciar = self.request.query_params.get('pode_iniciar', None)
        if pode_iniciar and pode_iniciar.lower() == 'true':
            # Filtra tarefas sem dependências ou com todas as dependências concluídas
            queryset = queryset.filter(
                models.Q(dependencias__isnull=True) |
                ~models.Q(dependencias__status__in=['nao_iniciada', 'em_andamento', 'pausada'])
            ).distinct()
            
        return queryset

    @action(detail=True, methods=['patch'])
    def update_status(self, request, pk=None):
        """
        Atualiza o status da tarefa.
        """
        task = self.get_object()
        serializer = TaskUpdateStatusSerializer(task, data=request.data, partial=True)
        
        if serializer.is_valid():
            serializer.save()
            return Response(TaskSerializer(task).data)
        
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    @action(detail=True, methods=['post'])
    def time_control(self, request, pk=None):
        """
        Controla o tempo de trabalho da tarefa (iniciar/pausar/concluir).
        """
        task = self.get_object()
        serializer = TaskTimeControlSerializer(data=request.data, context={'task': task})
        
        if serializer.is_valid():
            action_type = serializer.validated_data['action']
            
            if action_type == 'iniciar':
                task.iniciar_trabalho()
            elif action_type == 'pausar':
                task.pausar_trabalho()
            elif action_type == 'concluir':
                task.concluir_tarefa()
                
                # Criar notificação de tarefa concluída
                from notification.models import Notification
                if task.responsavel:
                    Notification.criar_notificacao_tarefa_concluida(task)
            
            return Response(TaskSerializer(task).data)
        
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    @action(detail=True, methods=['post'])
    def assign_responsible(self, request, pk=None):
        """
        Atribui um responsável à tarefa.
        """
        task = self.get_object()
        responsavel_id = request.data.get('responsavel_id')
        
        if responsavel_id:
            from collaborator.models import Collaborator
            try:
                responsavel = Collaborator.objects.get(id=responsavel_id)
                old_responsavel = task.responsavel
                
                task.responsavel = responsavel
                task.save()
                
                # Criar notificação se o responsável mudou
                if old_responsavel != responsavel:
                    from notification.models import Notification
                    Notification.criar_notificacao_responsavel_adicionado(task, responsavel)
                
                return Response(TaskSerializer(task).data)
                
            except Collaborator.DoesNotExist:
                return Response(
                    {"error": "Colaborador não encontrado"}, 
                    status=status.HTTP_404_NOT_FOUND
                )
        
        return Response(
            {"error": "Campo 'responsavel_id' é obrigatório"}, 
            status=status.HTTP_400_BAD_REQUEST
        )

    @action(detail=True, methods=['post'])
    def add_dependency(self, request, pk=None):
        """
        Adiciona uma dependência à tarefa.
        """
        task = self.get_object()
        dependency_id = request.data.get('dependency_id')
        
        if dependency_id:
            try:
                dependency = Task.objects.get(id=dependency_id)
                
                # Verifica se não cria dependência circular
                if dependency == task:
                    return Response(
                        {"error": "Uma tarefa não pode depender de si mesma"}, 
                        status=status.HTTP_400_BAD_REQUEST
                    )
                
                # Verifica se a dependência é do mesmo projeto
                if dependency.group.board.project != task.group.board.project:
                    return Response(
                        {"error": "Dependências devem ser do mesmo projeto"}, 
                        status=status.HTTP_400_BAD_REQUEST
                    )
                
                task.dependencias.add(dependency)
                task.recalcular_datas()
                
                return Response(TaskSerializer(task).data)
                
            except Task.DoesNotExist:
                return Response(
                    {"error": "Tarefa de dependência não encontrada"}, 
                    status=status.HTTP_404_NOT_FOUND
                )
        
        return Response(
            {"error": "Campo 'dependency_id' é obrigatório"}, 
            status=status.HTTP_400_BAD_REQUEST
        )

    @action(detail=True, methods=['delete'])
    def remove_dependency(self, request, pk=None):
        """
        Remove uma dependência da tarefa.
        """
        task = self.get_object()
        dependency_id = request.data.get('dependency_id')
        
        if dependency_id:
            try:
                dependency = Task.objects.get(id=dependency_id)
                task.dependencias.remove(dependency)
                
                return Response(TaskSerializer(task).data)
                
            except Task.DoesNotExist:
                return Response(
                    {"error": "Tarefa de dependência não encontrada"}, 
                    status=status.HTTP_404_NOT_FOUND
                )
        
        return Response(
            {"error": "Campo 'dependency_id' é obrigatório"}, 
            status=status.HTTP_400_BAD_REQUEST
        )

    @action(detail=False, methods=['get'])
    def my_tasks(self, request):
        """
        Retorna as tarefas do usuário logado.
        """
        try:
            collaborator = request.user.collaborator
            tasks = Task.objects.filter(responsavel=collaborator)
            
            serializer = TaskListSerializer(tasks, many=True)
            return Response(serializer.data)
            
        except:
            return Response(
                {"error": "Colaborador não encontrado para este usuário"}, 
                status=status.HTTP_404_NOT_FOUND
            )

