from django import forms
from .models import Task, Group
from collaborator.models import Collaborator
from board.models import Board


class GroupForm(forms.ModelForm):
    """
    Formulário para criação e edição de grupos.
    """
    
    class Meta:
        model = Group
        fields = ['nome', 'board', 'tipo', 'descricao', 'cor', 'ordem', 'limite_tarefas', 'ativo']
        widgets = {
            'nome': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': 'Nome do grupo'
            }),
            'board': forms.Select(attrs={
                'class': 'form-select'
            }),
            'tipo': forms.Select(attrs={
                'class': 'form-select'
            }),
            'descricao': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 3,
                'placeholder': 'Descrição do grupo'
            }),
            'cor': forms.TextInput(attrs={
                'class': 'form-control',
                'type': 'color'
            }),
            'ordem': forms.NumberInput(attrs={
                'class': 'form-control',
                'min': 0
            }),
            'limite_tarefas': forms.NumberInput(attrs={
                'class': 'form-control',
                'min': 1
            }),
            'ativo': forms.CheckboxInput(attrs={
                'class': 'form-check-input'
            })
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        
        # Filtrar apenas boards ativos
        self.fields['board'].queryset = Board.objects.filter(ativo=True).order_by('nome')
        
        # Adiciona classes CSS aos labels
        for field_name, field in self.fields.items():
            if field_name != 'ativo':
                field.widget.attrs.update({'class': field.widget.attrs.get('class', '') + ' form-control'})


class TaskCreateForm(forms.ModelForm):
    """
    Formulário para criação de tarefas (campos obrigatórios simplificados).
    """
    
    class Meta:
        model = Task
        fields = ['nome', 'descricao', 'group', 'responsavel', 'prioridade', 'status', 'cronograma_inicio', 'cronograma_fim']
        widgets = {
            'nome': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': 'Nome da tarefa'
            }),
            'descricao': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 3,
                'placeholder': 'Descrição da tarefa'
            }),
            'group': forms.Select(attrs={
                'class': 'form-select'
            }),
            'responsavel': forms.Select(attrs={
                'class': 'form-select'
            }),
            'prioridade': forms.Select(attrs={
                'class': 'form-select'
            }),
            'status': forms.Select(attrs={
                'class': 'form-select'
            }),
            'cronograma_inicio': forms.DateInput(attrs={
                'class': 'form-control',
                'type': 'date'
            }),
            'cronograma_fim': forms.DateInput(attrs={
                'class': 'form-control',
                'type': 'date'
            })
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        
        # Filtrar apenas groups ativos
        self.fields['group'].queryset = Group.objects.filter(ativo=True).order_by('nome')
        
        # Filtrar apenas colaboradores ativos
        self.fields['responsavel'].queryset = Collaborator.objects.filter(ativo=True).order_by('nome')
        
        # Adiciona classes CSS aos labels
        for field_name, field in self.fields.items():
            field.widget.attrs.update({'class': field.widget.attrs.get('class', '') + ' form-control'})


class TaskForm(forms.ModelForm):
    """
    Formulário para criação e edição de tarefas (completo).
    """
    
    class Meta:
        model = Task
        fields = ['nome', 'descricao', 'group', 'responsavel', 'prioridade', 'status', 'cronograma_inicio', 'cronograma_fim', 'duracao_dias', 'esforco_previsto', 'esforco_utilizado']
        widgets = {
            'nome': forms.TextInput(attrs={
                'class': 'form-control',
                'placeholder': 'Nome da tarefa'
            }),
            'descricao': forms.Textarea(attrs={
                'class': 'form-control',
                'rows': 3,
                'placeholder': 'Descrição da tarefa'
            }),
            'group': forms.Select(attrs={
                'class': 'form-select'
            }),
            'responsavel': forms.Select(attrs={
                'class': 'form-select'
            }),
            'prioridade': forms.Select(attrs={
                'class': 'form-select'
            }),
            'status': forms.Select(attrs={
                'class': 'form-select'
            }),
            'cronograma_inicio': forms.DateInput(attrs={
                'class': 'form-control',
                'type': 'date'
            }),
            'cronograma_fim': forms.DateInput(attrs={
                'class': 'form-control',
                'type': 'date'
            }),
            'duracao_dias': forms.NumberInput(attrs={
                'class': 'form-control',
                'type': 'number',
                'min': '1',
                'placeholder': '1'
            }),
            'esforco_previsto': forms.NumberInput(attrs={
                'class': 'form-control',
                'type': 'number',
                'step': '0.01',
                'min': '0',
                'placeholder': '0.00'
            }),
            'esforco_utilizado': forms.NumberInput(attrs={
                'class': 'form-control',
                'type': 'number',
                'step': '0.01',
                'min': '0',
                'placeholder': '0.00',
                'readonly': 'readonly'
            })
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        
        # Filtrar apenas groups ativos
        self.fields['group'].queryset = Group.objects.filter(ativo=True).order_by('nome')
        
        # Filtrar apenas colaboradores ativos
        self.fields['responsavel'].queryset = Collaborator.objects.filter(ativo=True).order_by('nome')
        
        # Adiciona classes CSS aos labels
        for field_name, field in self.fields.items():
            field.widget.attrs.update({'class': field.widget.attrs.get('class', '') + ' form-control'}) 