from django.db import models
from django.core.validators import MinValueValidator, MaxValueValidator
from django.utils import timezone
from datetime import timedelta
from board.models import Board
from collaborator.models import Collaborator


class Group(models.Model):
    """
    Modelo para representar grupos de tarefas.
    Um grupo organiza tarefas relacionadas dentro de um board.
    """
    
    TIPO_CHOICES = [
        ('backlog', 'Backlog'),
        ('planejamento', 'Planejamento'),
        ('desenvolvimento', 'Desenvolvimento'),
        ('teste', 'Teste'),
        ('revisao', 'Revisão'),
        ('producao', 'Produção'),
        ('concluido', 'Concluído'),
        ('arquivado', 'Arquivado'),
    ]
    
    nome = models.CharField(
        max_length=200,
        verbose_name="Nome",
        help_text="Nome do grupo"
    )
    
    board = models.ForeignKey(
        Board,
        on_delete=models.CASCADE,
        verbose_name="Quadro",
        help_text="Quadro ao qual o grupo pertence",
        related_name="groups"
    )
    
    tipo = models.CharField(
        max_length=20,
        choices=TIPO_CHOICES,
        default='desenvolvimento',
        verbose_name="Tipo",
        help_text="Tipo do grupo (define o fluxo de trabalho)"
    )
    
    descricao = models.TextField(
        blank=True,
        null=True,
        verbose_name="Descrição",
        help_text="Descrição do grupo"
    )
    
    cor = models.CharField(
        max_length=7,
        default="#6c757d",
        verbose_name="Cor",
        help_text="Cor do grupo (formato hexadecimal)"
    )
    
    ordem = models.IntegerField(
        default=0,
        verbose_name="Ordem",
        help_text="Ordem de exibição do grupo"
    )
    
    limite_tarefas = models.IntegerField(
        blank=True,
        null=True,
        verbose_name="Limite de Tarefas",
        help_text="Número máximo de tarefas permitidas no grupo (WIP limit)"
    )
    
    ativo = models.BooleanField(
        default=True,
        verbose_name="Ativo",
        help_text="Indica se o grupo está ativo"
    )
    
    # Campos de controle de tempo
    tempo_estimado_total = models.DecimalField(
        max_digits=8,
        decimal_places=2,
        default=0,
        validators=[MinValueValidator(0)],
        verbose_name="Tempo Estimado Total (horas)",
        help_text="Soma do tempo estimado de todas as tarefas"
    )
    
    tempo_utilizado_total = models.DecimalField(
        max_digits=8,
        decimal_places=2,
        default=0,
        validators=[MinValueValidator(0)],
        verbose_name="Tempo Utilizado Total (horas)",
        help_text="Soma do tempo utilizado de todas as tarefas"
    )
    
    # Campos de cronograma
    data_inicio_prevista = models.DateField(
        blank=True,
        null=True,
        verbose_name="Data de Início Prevista",
        help_text="Data de início prevista do grupo"
    )
    
    data_fim_prevista = models.DateField(
        blank=True,
        null=True,
        verbose_name="Data de Fim Prevista",
        help_text="Data de fim prevista do grupo"
    )
    
    data_inicio_real = models.DateField(
        blank=True,
        null=True,
        verbose_name="Data de Início Real",
        help_text="Data real de início do grupo"
    )
    
    data_fim_real = models.DateField(
        blank=True,
        null=True,
        verbose_name="Data de Fim Real",
        help_text="Data real de fim do grupo"
    )
    
    # Campos de auditoria
    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name="Criado em"
    )
    
    updated_at = models.DateTimeField(
        auto_now=True,
        verbose_name="Atualizado em"
    )

    class Meta:
        verbose_name = "Grupo"
        verbose_name_plural = "Grupos"
        ordering = ['board', 'ordem', 'nome']
        unique_together = ['board', 'nome']

    def __str__(self):
        return f"{self.board.nome} - {self.nome}"

    def get_absolute_url(self):
        from django.urls import reverse
        return reverse('task:group_detail', kwargs={'pk': self.pk})

    @property
    def total_tasks(self):
        return self.tasks.count()

    @property
    def tasks_concluidas(self):
        return self.tasks.filter(status='concluida').count()

    @property
    def tasks_em_andamento(self):
        return self.tasks.filter(status='em_andamento').count()

    @property
    def tasks_pausadas(self):
        return self.tasks.filter(status='pausada').count()

    @property
    def tasks_nao_iniciadas(self):
        return self.tasks.filter(status='nao_iniciada').count()

    @property
    def progresso_percentual(self):
        """Calcula o progresso percentual do grupo baseado nas tarefas."""
        total = self.total_tasks
        if total == 0:
            return 0
        concluidas = self.tasks_concluidas
        return round((concluidas / total) * 100, 2)

    @property
    def tempo_estimado_atual(self):
        """Calcula o tempo estimado total das tarefas ativas."""
        return self.tasks.aggregate(
            total=models.Sum('esforco_previsto')
        )['total'] or 0

    @property
    def tempo_utilizado_atual(self):
        """Calcula o tempo utilizado total das tarefas."""
        return self.tasks.aggregate(
            total=models.Sum('esforco_utilizado')
        )['total'] or 0

    @property
    def is_atrasado(self):
        """Verifica se o grupo está atrasado."""
        if not self.data_fim_prevista:
            return False
        return timezone.now().date() > self.data_fim_prevista and self.progresso_percentual < 100

    @property
    def dias_atraso(self):
        """Retorna o número de dias de atraso."""
        if not self.is_atrasado:
            return 0
        return (timezone.now().date() - self.data_fim_prevista).days

    @property
    def is_limite_atingido(self):
        """Verifica se o limite de tarefas foi atingido."""
        if not self.limite_tarefas:
            return False
        return self.total_tasks >= self.limite_tarefas

    @property
    def tarefas_por_prioridade(self):
        """Retorna estatísticas de tarefas por prioridade."""
        return {
            'alta': self.tasks.filter(prioridade='alta').count(),
            'media': self.tasks.filter(prioridade='media').count(),
            'baixa': self.tasks.filter(prioridade='baixa').count(),
        }

    @property
    def tarefas_por_status(self):
        """Retorna estatísticas de tarefas por status."""
        return {
            'nao_iniciada': self.tasks_nao_iniciadas,
            'em_andamento': self.tasks_em_andamento,
            'pausada': self.tasks_pausadas,
            'concluida': self.tasks_concluidas,
            'cancelada': self.tasks.filter(status='cancelada').count(),
        }

    def recalcular_tempos(self):
        """Recalcula os tempos totais do grupo."""
        # Calcular tempo estimado total
        tempo_estimado = self.tasks.aggregate(
            total=models.Sum('esforco_previsto')
        )['total'] or 0
        
        # Calcular tempo utilizado total
        tempo_utilizado = self.tasks.aggregate(
            total=models.Sum('esforco_utilizado')
        )['total'] or 0
        
        # Atualizar campos diretamente sem chamar save()
        self.tempo_estimado_total = tempo_estimado
        self.tempo_utilizado_total = tempo_utilizado

    def recalcular_datas(self):
        """Recalcula as datas do grupo baseado nas tarefas."""
        tasks = self.tasks.all()
        if tasks.exists():
            # Data de início: menor data de início das tarefas
            min_inicio = tasks.aggregate(
                min_inicio=models.Min('cronograma_inicio')
            )['min_inicio']
            if min_inicio:
                self.data_inicio_prevista = min_inicio

            # Data de fim: maior data de fim das tarefas
            max_fim = tasks.aggregate(
                max_fim=models.Max('cronograma_fim')
            )['max_fim']
            if max_fim:
                self.data_fim_prevista = max_fim

            self.save(update_fields=['data_inicio_prevista', 'data_fim_prevista'])

    def pode_receber_tarefa(self):
        """Verifica se o grupo pode receber uma nova tarefa."""
        if not self.ativo:
            return False
        if self.is_limite_atingido:
            return False
        return True

    def get_tarefas_ordenadas(self):
        """Retorna as tarefas ordenadas por prioridade e data."""
        return self.tasks.order_by(
            '-prioridade',  # Alta, Média, Baixa
            'cronograma_inicio',
            'nome'
        )

    def save(self, *args, **kwargs):
        # Auto-calcular datas se não definidas e se já tem ID (não é criação)
        if self.pk and (not self.data_inicio_prevista or not self.data_fim_prevista):
            self.recalcular_datas()
        
        # Auto-calcular tempos se já tem ID (não é criação)
        if self.pk:
            self.recalcular_tempos()
        
        super().save(*args, **kwargs)


class Task(models.Model):
    """
    Modelo para representar tarefas.
    Uma tarefa é a unidade básica de trabalho no sistema.
    """
    
    STATUS_CHOICES = [
        ('nao_iniciada', 'Não Iniciada'),
        ('em_andamento', 'Em Andamento'),
        ('pausada', 'Pausada'),
        ('concluida', 'Concluída'),
        ('cancelada', 'Cancelada'),
    ]
    
    PRIORIDADE_CHOICES = [
        ('baixa', 'Baixa'),
        ('media', 'Média'),
        ('alta', 'Alta'),
    ]
    
    nome = models.CharField(
        max_length=200,
        verbose_name="Nome",
        help_text="Nome da tarefa"
    )
    
    group = models.ForeignKey(
        Group,
        on_delete=models.CASCADE,
        verbose_name="Grupo",
        help_text="Grupo ao qual a tarefa pertence",
        related_name="tasks"
    )
    
    descricao = models.TextField(
        blank=True,
        null=True,
        verbose_name="Descrição",
        help_text="Descrição detalhada da tarefa"
    )
    
    status = models.CharField(
        max_length=20,
        choices=STATUS_CHOICES,
        default='nao_iniciada',
        verbose_name="Status",
        help_text="Status atual da tarefa"
    )
    
    prioridade = models.CharField(
        max_length=10,
        choices=PRIORIDADE_CHOICES,
        default='media',
        verbose_name="Prioridade",
        help_text="Prioridade da tarefa"
    )
    
    cronograma_inicio = models.DateField(
        verbose_name="Data de Início",
        help_text="Data de início planejada da tarefa"
    )
    
    cronograma_fim = models.DateField(
        verbose_name="Data de Fim",
        help_text="Data de fim planejada da tarefa"
    )
    
    duracao_dias = models.IntegerField(
        validators=[MinValueValidator(1)],
        verbose_name="Duração (dias)",
        help_text="Duração da tarefa em dias úteis"
    )
    
    esforco_previsto = models.DecimalField(
        max_digits=6,
        decimal_places=2,
        validators=[MinValueValidator(0)],
        verbose_name="Esforço Previsto (horas)",
        help_text="Esforço previsto em horas"
    )
    
    esforco_utilizado = models.DecimalField(
        max_digits=6,
        decimal_places=2,
        default=0,
        validators=[MinValueValidator(0)],
        verbose_name="Esforço Utilizado (horas)",
        help_text="Esforço já utilizado em horas"
    )
    
    responsavel = models.ForeignKey(
        Collaborator,
        on_delete=models.SET_NULL,
        blank=True,
        null=True,
        verbose_name="Responsável",
        help_text="Colaborador responsável pela tarefa",
        related_name="tasks"
    )
    
    data_conclusao = models.DateTimeField(
        blank=True,
        null=True,
        verbose_name="Data de Conclusão",
        help_text="Data e hora de conclusão da tarefa"
    )
    
    dependencias = models.ManyToManyField(
        'self',
        blank=True,
        symmetrical=False,
        verbose_name="Dependências",
        help_text="Tarefas das quais esta tarefa depende",
        related_name="dependentes"
    )
    
    # Campos de controle de tempo
    tempo_iniciado = models.DateTimeField(
        blank=True,
        null=True,
        verbose_name="Tempo Iniciado",
        help_text="Timestamp do último início de trabalho"
    )
    
    tempo_pausado = models.DateTimeField(
        blank=True,
        null=True,
        verbose_name="Tempo Pausado",
        help_text="Timestamp da última pausa"
    )
    
    tempo_total_trabalhado = models.DurationField(
        default=timedelta(0),
        verbose_name="Tempo Total Trabalhado",
        help_text="Tempo total trabalhado na tarefa"
    )
    
    # Campo para armazenar logs de sessões
    logs_sessoes = models.JSONField(
        default=list,
        blank=True,
        verbose_name="Logs de Sessões",
        help_text="Histórico de sessões de trabalho"
    )
    
    # Campos de auditoria
    inserted_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name="Inserido em"
    )
    
    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name="Criado em"
    )
    
    updated_at = models.DateTimeField(
        auto_now=True,
        verbose_name="Atualizado em"
    )

    class Meta:
        verbose_name = "Tarefa"
        verbose_name_plural = "Tarefas"
        ordering = ['group', 'cronograma_inicio', 'nome']

    def __str__(self):
        return f"{self.group.nome} - {self.nome}"

    def get_absolute_url(self):
        from django.urls import reverse
        return reverse('task:detail', kwargs={'pk': self.pk})

    @property
    def is_atrasada(self):
        """Verifica se a tarefa está atrasada."""
        if self.status in ['concluida', 'cancelada']:
            return False
        return timezone.now().date() > self.cronograma_fim

    @property
    def dias_atraso(self):
        """Retorna o número de dias de atraso."""
        if not self.is_atrasada:
            return 0
        return (timezone.now().date() - self.cronograma_fim).days

    @property
    def progresso_percentual(self):
        """Calcula o progresso baseado no esforço utilizado."""
        if self.esforco_previsto == 0:
            return 0
        progresso = (self.esforco_utilizado / self.esforco_previsto) * 100
        return float(min(progresso, 100))  # Máximo 100% e garantir que é float

    @property
    def pode_iniciar(self):
        """Verifica se a tarefa pode ser iniciada (dependências concluídas)."""
        return all(dep.status == 'concluida' for dep in self.dependencias.all())

    @property
    def tempo_trabalhado_hoje(self):
        """Retorna o tempo trabalhado hoje."""
        # Calcular tempo das sessões salvas hoje
        tempo_sessoes = self.obter_tempo_por_data(timezone.now().date())
        
        # Adicionar tempo da sessão atual se estiver trabalhando hoje
        if self.tempo_iniciado and self.tempo_iniciado.date() == timezone.now().date():
            if self.status == 'em_andamento':
                # Se está em andamento, adicionar o tempo atual
                tempo_sessoes += timezone.now() - self.tempo_iniciado
            # Se está pausada, o tempo já foi adicionado aos logs quando pausou
        
        return tempo_sessoes

    @property
    def tempo_atual_sessao(self):
        """Retorna o tempo da sessão atual (se estiver em andamento)."""
        if self.tempo_iniciado and not self.tempo_pausado and self.status == 'em_andamento':
            return timezone.now() - self.tempo_iniciado
        return timedelta(0)
    
    @property
    def tempo_sessao_formatado(self):
        """Retorna o tempo da sessão atual formatado."""
        tempo = self.tempo_atual_sessao
        horas = int(tempo.total_seconds() // 3600)
        minutos = int((tempo.total_seconds() % 3600) // 60)
        segundos = int(tempo.total_seconds() % 60)
        return f"{horas:02d}:{minutos:02d}:{segundos:02d}"

    @property
    def tempo_total_formatado(self):
        """Retorna o tempo total trabalhado formatado."""
        total = self.tempo_total_trabalhado + self.tempo_atual_sessao
        horas = int(total.total_seconds() // 3600)
        minutos = int((total.total_seconds() % 3600) // 60)
        segundos = int(total.total_seconds() % 60)
        return f"{horas:02d}:{minutos:02d}:{segundos:02d}"

    @property
    def tempo_hoje_formatado(self):
        """Retorna o tempo trabalhado hoje formatado."""
        tempo = self.tempo_trabalhado_hoje
        horas = int(tempo.total_seconds() // 3600)
        minutos = int((tempo.total_seconds() % 3600) // 60)
        segundos = int(tempo.total_seconds() % 60)
        return f"{horas:02d}:{minutos:02d}:{segundos:02d}"

    @property
    def esta_trabalhando(self):
        """Verifica se a tarefa está sendo trabalhada no momento."""
        return self.tempo_iniciado and not self.tempo_pausado and self.status == 'em_andamento'

    def iniciar_trabalho(self):
        """Inicia o cronômetro de trabalho."""
        if self.status == 'nao_iniciada':
            self.status = 'em_andamento'
            self.tempo_iniciado = timezone.now()
            self.tempo_pausado = None
        elif self.status == 'pausada':
            self.status = 'em_andamento'
            # Retomar de onde parou
            if self.tempo_pausado and self.tempo_iniciado:
                tempo_ja_trabalhado = self.tempo_pausado - self.tempo_iniciado
                self.tempo_iniciado = timezone.now() - tempo_ja_trabalhado
            else:
                self.tempo_iniciado = timezone.now()
            self.tempo_pausado = None
        elif self.status == 'em_andamento':
            # Se já está em andamento mas não está sendo trabalhada ativamente
            if not self.esta_trabalhando:
                self.tempo_iniciado = timezone.now()
                self.tempo_pausado = None
            else:
                raise ValueError("Tarefa já está sendo trabalhada")
        elif self.status in ['concluida', 'cancelada']:
            raise ValueError(f"Tarefa não pode ser iniciada no status '{self.get_status_display()}'")
        else:
            raise ValueError(f"Status '{self.status}' não suportado para iniciar trabalho")
        
        self.save()

    def pausar_trabalho(self):
        """Pausa o cronômetro de trabalho."""
        if self.tempo_iniciado and not self.tempo_pausado and self.status == 'em_andamento':
            self.tempo_pausado = timezone.now()
            tempo_sessao = self.tempo_pausado - self.tempo_iniciado
            self.tempo_total_trabalhado += tempo_sessao
            
            # Converte para horas e atualiza esforço utilizado
            horas_sessao = tempo_sessao.total_seconds() / 3600
            from decimal import Decimal
            self.esforco_utilizado += Decimal(str(horas_sessao))
            
            # Salvar log da sessão
            self.adicionar_log_sessao(self.tempo_iniciado, self.tempo_pausado, tempo_sessao)
            
            self.status = 'pausada'
            self.save()
        else:
            raise ValueError("Tarefa não está em andamento ou já está pausada")

    def retomar_trabalho(self):
        """Retoma o trabalho pausado."""
        if self.status == 'pausada':
            if self.tempo_pausado and self.tempo_iniciado:
                self.status = 'em_andamento'
                # Definir novo tempo_iniciado para a nova sessão
                # O tempo anterior já foi salvo no tempo_total_trabalhado quando pausou
                self.tempo_iniciado = timezone.now()
                self.tempo_pausado = None
                self.save()
            else:
                # Se está pausada mas não tem tempo pausado, iniciar normalmente
                self.iniciar_trabalho()
        else:
            raise ValueError(f"Tarefa não está pausada (status atual: {self.get_status_display()})")





    def concluir_tarefa(self):
        """Marca a tarefa como concluída."""
        if self.tempo_iniciado and not self.tempo_pausado:
            self.pausar_trabalho()
        
        self.status = 'concluida'
        self.data_conclusao = timezone.now()
        self.save()
        
        # Atualiza dependentes
        self.atualizar_dependentes()

    def atualizar_dependentes(self):
        """Atualiza as datas dos dependentes quando a tarefa é concluída."""
        for dependente in self.dependentes.all():
            if dependente.pode_iniciar and dependente.status == 'nao_iniciada':
                # Lógica para atualizar datas em cascata
                dependente.recalcular_datas()

    def recalcular_datas(self):
        """Recalcula as datas da tarefa baseado nas dependências."""
        if self.dependencias.exists():
            ultima_data_dep = max(dep.cronograma_fim for dep in self.dependencias.all())
            # Adiciona 1 dia útil após a última dependência
            self.cronograma_inicio = ultima_data_dep + timedelta(days=1)
            self.cronograma_fim = self.cronograma_inicio + timedelta(days=self.duracao_dias - 1)
            self.save()

    def adicionar_log_sessao(self, inicio, fim, duracao):
        """Adiciona um log de sessão ao histórico."""
        if not self.logs_sessoes:
            self.logs_sessoes = []
        
        log = {
            'id': len(self.logs_sessoes) + 1,
            'inicio': inicio.isoformat() if inicio else None,
            'fim': fim.isoformat() if fim else None,
            'duracao_segundos': duracao.total_seconds(),
            'duracao_formatada': self.formatar_duracao(duracao),
            'data': timezone.now().date().isoformat(),  # Sempre usar data atual
            'criado_em': timezone.now().isoformat()
        }
        
        self.logs_sessoes.append(log)
    
    def formatar_duracao(self, duracao):
        """Formata uma duração em formato legível."""
        horas = int(duracao.total_seconds() // 3600)
        minutos = int((duracao.total_seconds() % 3600) // 60)
        segundos = int(duracao.total_seconds() % 60)
        return f"{horas:02d}:{minutos:02d}:{segundos:02d}"
    
    def obter_logs_por_data(self, data=None):
        """Retorna logs de sessões por data específica."""
        from datetime import datetime
        
        if not data:
            data = timezone.now().date()
        
        if isinstance(data, str):
            data = datetime.strptime(data, '%Y-%m-%d').date()
        
        logs_filtrados = []
        for log in self.logs_sessoes:
            log_data = datetime.strptime(log['data'], '%Y-%m-%d').date()
            if log_data == data:
                logs_filtrados.append(log)
        
        return logs_filtrados
    
    def obter_tempo_por_data(self, data=None):
        """Retorna o tempo total trabalhado em uma data específica."""
        from datetime import timedelta
        
        logs = self.obter_logs_por_data(data)
        tempo_total = timedelta(0)
        
        for log in logs:
            tempo_total += timedelta(seconds=log['duracao_segundos'])
        
        return tempo_total
    
    def editar_log_sessao(self, log_id, nova_duracao_segundos, nova_data=None):
        """Edita um log de sessão específico."""
        from datetime import timedelta
        
        for log in self.logs_sessoes:
            if log['id'] == log_id:
                # Calcular diferença de tempo
                duracao_antiga = timedelta(seconds=log['duracao_segundos'])
                duracao_nova = timedelta(seconds=nova_duracao_segundos)
                diferenca = duracao_nova - duracao_antiga
                
                # Atualizar log
                log['duracao_segundos'] = nova_duracao_segundos
                log['duracao_formatada'] = self.formatar_duracao(duracao_nova)
                
                if nova_data:
                    log['data'] = nova_data
                
                # Atualizar tempo total
                self.tempo_total_trabalhado += diferenca
                
                # Atualizar esforço utilizado
                horas_diferenca = diferenca.total_seconds() / 3600
                from decimal import Decimal
                self.esforco_utilizado += Decimal(str(horas_diferenca))
                
                return True
        
        return False
    
    def save(self, *args, **kwargs):
        from datetime import datetime
        
        # Converter duracao_dias para inteiro se for string
        if isinstance(self.duracao_dias, str):
            try:
                self.duracao_dias = int(self.duracao_dias)
            except (ValueError, TypeError):
                pass
        
        # Converter cronograma_inicio para date se for string
        if isinstance(self.cronograma_inicio, str):
            try:
                self.cronograma_inicio = datetime.strptime(self.cronograma_inicio, '%Y-%m-%d').date()
            except (ValueError, TypeError):
                pass
        
        # Validação: cronograma_fim deve ser consistente com duração
        if self.cronograma_inicio and self.duracao_dias:
            calculated_fim = self.cronograma_inicio + timedelta(days=self.duracao_dias - 1)
            if not self.cronograma_fim or self.cronograma_fim != calculated_fim:
                self.cronograma_fim = calculated_fim
        
        # Auto-set data_conclusao quando status muda para concluida
        if self.status == 'concluida' and not self.data_conclusao:
            self.data_conclusao = timezone.now()
        elif self.status != 'concluida':
            self.data_conclusao = None
            
        super().save(*args, **kwargs)

