from rest_framework import serializers
from .models import Group, Task
from board.serializers import BoardListSerializer
from collaborator.serializers import CollaboratorListSerializer


class GroupSerializer(serializers.ModelSerializer):
    """
    Serializer para o modelo Group.
    """
    
    board_nome = serializers.CharField(source='board.nome', read_only=True)
    total_tasks = serializers.ReadOnlyField()
    tasks_concluidas = serializers.ReadOnlyField()
    progresso_percentual = serializers.ReadOnlyField()
    
    class Meta:
        model = Group
        fields = [
            'id', 'nome', 'board', 'board_nome', 'descricao', 'cor', 'ordem', 'ativo',
            'total_tasks', 'tasks_concluidas', 'progresso_percentual',
            'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'created_at', 'updated_at']


class GroupDetailSerializer(serializers.ModelSerializer):
    """
    Serializer detalhado para o modelo Group.
    """
    
    board = BoardListSerializer(read_only=True)
    tasks = serializers.SerializerMethodField()
    progresso_percentual = serializers.ReadOnlyField()
    
    class Meta:
        model = Group
        fields = [
            'id', 'nome', 'board', 'descricao', 'cor', 'ordem', 'ativo',
            'tasks', 'progresso_percentual',
            'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'created_at', 'updated_at']
    
    def get_tasks(self, obj):
        tasks = obj.tasks.all().order_by('cronograma_inicio', 'nome')
        return TaskListSerializer(tasks, many=True).data


class GroupListSerializer(serializers.ModelSerializer):
    """
    Serializer simplificado para listagem de grupos.
    """
    
    board_nome = serializers.CharField(source='board.nome', read_only=True)
    total_tasks = serializers.ReadOnlyField()
    progresso_percentual = serializers.ReadOnlyField()
    
    class Meta:
        model = Group
        fields = [
            'id', 'nome', 'board_nome', 'cor', 'ordem', 'ativo',
            'total_tasks', 'progresso_percentual'
        ]


class TaskSerializer(serializers.ModelSerializer):
    """
    Serializer para o modelo Task.
    """
    
    group_nome = serializers.CharField(source='group.nome', read_only=True)
    board_nome = serializers.CharField(source='group.board.nome', read_only=True)
    project_nome = serializers.CharField(source='group.board.project.nome', read_only=True)
    responsavel_nome = serializers.CharField(source='responsavel.nome', read_only=True)
    is_atrasada = serializers.ReadOnlyField()
    dias_atraso = serializers.ReadOnlyField()
    progresso_percentual = serializers.ReadOnlyField()
    pode_iniciar = serializers.ReadOnlyField()
    tempo_trabalhado_hoje = serializers.ReadOnlyField()
    
    class Meta:
        model = Task
        fields = [
            'id', 'nome', 'group', 'group_nome', 'board_nome', 'project_nome',
            'descricao', 'status', 'prioridade', 'cronograma_inicio', 'cronograma_fim',
            'duracao_dias', 'esforco_previsto', 'esforco_utilizado', 'responsavel',
            'responsavel_nome', 'data_conclusao', 'dependencias',
            'tempo_iniciado', 'tempo_pausado', 'tempo_total_trabalhado',
            'is_atrasada', 'dias_atraso', 'progresso_percentual', 'pode_iniciar',
            'tempo_trabalhado_hoje', 'inserted_at', 'created_at', 'updated_at'
        ]
        read_only_fields = [
            'id', 'data_conclusao', 'tempo_total_trabalhado',
            'inserted_at', 'created_at', 'updated_at'
        ]


class TaskDetailSerializer(serializers.ModelSerializer):
    """
    Serializer detalhado para o modelo Task.
    """
    
    group = GroupListSerializer(read_only=True)
    responsavel = CollaboratorListSerializer(read_only=True)
    dependencias = serializers.SerializerMethodField()
    dependentes = serializers.SerializerMethodField()
    is_atrasada = serializers.ReadOnlyField()
    progresso_percentual = serializers.ReadOnlyField()
    pode_iniciar = serializers.ReadOnlyField()
    
    class Meta:
        model = Task
        fields = [
            'id', 'nome', 'group', 'descricao', 'status', 'prioridade',
            'cronograma_inicio', 'cronograma_fim', 'duracao_dias',
            'esforco_previsto', 'esforco_utilizado', 'responsavel', 'data_conclusao',
            'dependencias', 'dependentes', 'tempo_iniciado', 'tempo_pausado',
            'tempo_total_trabalhado', 'is_atrasada', 'progresso_percentual',
            'pode_iniciar', 'inserted_at', 'created_at', 'updated_at'
        ]
        read_only_fields = [
            'id', 'data_conclusao', 'tempo_total_trabalhado',
            'inserted_at', 'created_at', 'updated_at'
        ]
    
    def get_dependencias(self, obj):
        dependencias = obj.dependencias.all()
        return TaskListSerializer(dependencias, many=True).data
    
    def get_dependentes(self, obj):
        dependentes = obj.dependentes.all()
        return TaskListSerializer(dependentes, many=True).data


class TaskListSerializer(serializers.ModelSerializer):
    """
    Serializer simplificado para listagem de tarefas.
    """
    
    group_nome = serializers.CharField(source='group.nome', read_only=True)
    responsavel_nome = serializers.CharField(source='responsavel.nome', read_only=True)
    is_atrasada = serializers.ReadOnlyField()
    progresso_percentual = serializers.ReadOnlyField()
    
    class Meta:
        model = Task
        fields = [
            'id', 'nome', 'group_nome', 'status', 'prioridade',
            'cronograma_inicio', 'cronograma_fim', 'responsavel_nome',
            'is_atrasada', 'progresso_percentual'
        ]


class TaskCreateSerializer(serializers.ModelSerializer):
    """
    Serializer para criação de tarefas.
    """
    
    class Meta:
        model = Task
        fields = [
            'nome', 'group', 'descricao', 'status', 'prioridade',
            'cronograma_inicio', 'duracao_dias', 'esforco_previsto',
            'responsavel', 'dependencias'
        ]
    
    def validate(self, data):
        # Valida se as dependências são do mesmo projeto
        if 'dependencias' in data and data['dependencias']:
            group = data['group']
            project = group.board.project
            
            for dep in data['dependencias']:
                if dep.group.board.project != project:
                    raise serializers.ValidationError(
                        "Dependências devem ser do mesmo projeto."
                    )
        
        return data


class TaskUpdateStatusSerializer(serializers.ModelSerializer):
    """
    Serializer para atualização de status da tarefa.
    """
    
    class Meta:
        model = Task
        fields = ['status', 'esforco_utilizado']


class TaskTimeControlSerializer(serializers.Serializer):
    """
    Serializer para controle de tempo da tarefa.
    """
    
    action = serializers.ChoiceField(choices=['iniciar', 'pausar', 'concluir'])
    
    def validate_action(self, value):
        task = self.context.get('task')
        
        if value == 'iniciar':
            if task.status in ['concluida', 'cancelada']:
                raise serializers.ValidationError("Não é possível iniciar uma tarefa concluída ou cancelada.")
            if not task.pode_iniciar:
                raise serializers.ValidationError("Tarefa possui dependências não concluídas.")
        
        elif value == 'pausar':
            if task.status != 'em_andamento':
                raise serializers.ValidationError("Só é possível pausar tarefas em andamento.")
        
        elif value == 'concluir':
            if task.status in ['concluida', 'cancelada']:
                raise serializers.ValidationError("Tarefa já está concluída ou cancelada.")
        
        return value

